clear all;

% Monte Carlo simulations for random walk rate frontier
% MPM 2012-06-14, 2012-11-12

gammas = 0.25:0.05:1.5; %gamma values
numrep = 500; %Number of simulations
Ts = 200:100:700; %Array of T values
c = 5; %c value for RW: h_NT = c/T^gamma
r = 5; %Number of factors

rho = 0; %Common AR(1) coefficient for factors
a = 0; %AR(1) coefficient for idiosyncratic errors
beta = 0; %Cross-sectional correlation coefficient for idiosyncratic errors
R2_dom = [0 0.8]; %Interval on which the R^2 for factors is uniformly distributed

filename = 'raterw'; %Filename for .mat file (equal to '' if results shouldn't be saved)

tic; %Start timer
R2s_rw = zeros(length(gammas),numrep,length(Ts),2); %Array with simulation data

for ng = 1:length(gammas); %For each gamma value...
    gamma = gammas(ng);
    for n = 1:numrep; %For each repetition...
        for nT = 1:length(Ts); %For each T...
            T = Ts(nT); %T
            N = floor(T); %N

            %UNCOMMENT THE FOLLOWING IF BETA IS NOT 0
%             circul = reshape(mod(0:N^2-1,N+1),N,N);
%             uind = (triu(ones(N))==1);
%             circul_tp = circul';
%             circul(uind) = circul_tp(uind);
%             Omega = beta.^circul;
%             [OV,OD]=eig(Omega);
%             sqrtOmega = OV*sqrt(OD);

%             [F,y,~,e,Lambda_0] = genvars(T,N,r,rho,a,sqrtOmega,R2_dom); %Generate variables (beta not 0)

            [F,y,~,e,Lambda_0] = genvars(T,N,r,rho,a,eye(N),R2_dom); %Generate variables (beta=0)

            h = c/T^gamma; %h_NT
            zeta = randn(T,N*r);
            xi = filter(1,[1 -1],zeta);
            Lambda = repmat(Lambda_0(:)',T,1)+h*xi; %Array of lambda_t values

            R2s = summ_stats(F,Lambda_0,Lambda,y,e,0); %Retrieve trace R-square components
            R2s_rw(ng,n,nT,:) = R2s; %Store
        end;
    end;
    
    %Show progress
    fprintf('%4.2f\n',gamma);
    
end;
tstop=toc; %Stop timer
disp('Elapsed time');
disp(tstop);

%Calculate MSE statistic from trace R-square numerator and denominator
SE = mean(R2s_rw(:,:,:,2),2)-mean(R2s_rw(:,:,:,1),2);
SE = reshape(SE,length(gammas),length(Ts));
MSE = SE./repmat(Ts,length(gammas),1);

%Run regressions of log(MSE) on log(T)
rhat_rw = zeros(length(gammas),1);
for ng = 1:length(gammas);
    covm = cov(log(MSE(ng,:)),log(Ts));
    rhat_rw(ng) = covm(1,2)/covm(2,2);
end;

%Plot results and compare with theoretical frontier
figure;
theor = max(2-4*gammas,-1); %Theoretical frontier
plot(gammas,rhat_rw,'-xk',gammas,theor,'--k');
xlim([0.25 1.5]);
set(gca,'XTick',0.25:0.25:1.5);
title('Finite-sample and theoretical rate frontiers','FontSize',12);
xlabel('\gamma','FontSize',10);
ylabel('m','FontSize',10);
legend({'Finite-sample','Theoretical'},'Location','SouthEast');

if ~strcmp(filename,''); %If results are to be saved...
    save([filename '.mat']);
end;