/* ... Miscellaneous VAR Procedures ... */

proc(9) = varest(s,lag,tfirst,tlast,ivarmat,icomp);

/* Computes VAR and covariance matrix of estimated parameters
   
   Input:
   		    s = txn matrix of series
   		    lag = number of lags
   		    tfirst = index of first obs to use
   		    tlast = index of last obs to use
   		    ivarmat = 1 if covariance matrix of estimated parameters is computed
   		              0 otherwise 
   		    icomp = 1 Companion Form of VAR is computed
   		            0 otherwise 


   Output:
          beta = estimated VAR coefficients (including constant)
                 each column has coefficients for a single equation
          seps = innovation covariance matrix
          eps =  VAR residuals
          nobs = number of observations used for estimation
          ndf =  number of degrees of freedom in each equation 
          v_bs = covariance matrix of vec(beta)|vech(seps) ... 
                 computed if ivarmat = 1
          Companion Form Parameter, computed in icomp = 1;
          Q, M, G for model written in SS form
               y(t) = Q*z(t)
               z(t) = M*z(t-1) + G*u(t)
               var(u(t)) = I
*/

local ns, yv, x, i, tmp, beta, eps, seps, v_bs, v_beta, z, t, v_seps, cv_bs, m, q, g, b, comp, nobs, ndf, vhseps;
local trnd, trndpack, n1, n2;
@ Set Up VAR @

if tfirst .< lag+1; tfirst=lag+1; endif;
if tlast .> rows(s); tlast=rows(s); endif;
ns=cols(s);

yv=s[tfirst:tlast,.];
x=ones(rows(yv),1);
i=1; do while i<=lag;
  x=x~s[tfirst-i:tlast-i,.];
i=i+1; endo;

@ Eliminate Missing Values @
tmp=yv~x;
trnd=seqa(1,1,rows(tmp));
tmp=trnd~tmp;
tmp=packr(tmp);
trndpack=tmp[.,1];
tmp=tmp[.,2:cols(tmp)];
n1=trndpack[1]-1;  @ Number of periods lost in beginning @
n2=trnd[rows(trnd)]-trndpack[rows(trndpack)]; @ Number of periods lost at end @
yv=tmp[.,1:cols(yv)];
x=tmp[.,cols(yv)+1:cols(yv)+cols(x)];

@ Estimate VAR @
beta=invpd(x'x)*x'yv;
eps=yv-x*beta;
nobs=rows(x);
ndf=rows(x)-cols(x);
seps=(eps'eps)/ndf;


v_bs=0;
if ivarmat .== 1;
@ Calculate Covariance Matrix of Estimated VAR Parameters @
   @ --- Covariance Matrix of Vec(beta) --- @
     v_beta=seps.*.(invpd(x'x));

   @ --- Covariance Matrix of Vech(seps) --- @
     vhseps=vech(seps);
     z=zeros(rows(vhseps),rows(eps));
     t=1; do while t <= rows(eps);
      z[.,t] = vech(eps[t,.]'eps[t,.])-vhseps;
     t=t+1; endo;
     v_seps=z*z'/(cols(z)*cols(z));

   @ --- Covariance Between Vec(beta) and Vech(seps) --- @
    cv_bs = zeros(rows(v_beta),rows(v_seps));
     t=1; do while t <= cols(z);
      cv_bs = cv_bs + (eye(rows(seps)) .*. x[t,.]')*(eps[t,.]'.*. z[.,t]');
     t=t+1; endo;
      cv_bs = (eye(rows(seps)) .*. (inv(x'x)) )*cv_bs*(1.0/rows(eps));

  @ -- Put Results Together, Cov matrix of [Vec(beta)|Vech(seps)] --- @
   v_bs = (v_beta~cv_bs)|(cv_bs'~v_seps);
endif;



m=0; q=0; g=0;
if icomp .== 1;
 @
   Transform the VAR so that it is written in Standard form as:
   s(t)=P1*s(t-1) + P2*s(t-2) + ... + Pvarlag*s(t-varlag) + eps(t)
 @

 @ ---- Calculate Companion Matrix ---- @
 b=beta[2:rows(beta),.]';
 comp=zeros(cols(b),cols(b));
 comp[1:rows(b),.]=b;
 if cols(b) .> rows(b);
  comp[rows(b)+1:rows(comp),1:cols(comp)-rows(b)]=eye(cols(comp)-rows(b));
 endif;
 @ -- Write Model in SS Form --
     y(t) = Q*z(t)
     z(t) = M*z(t-1) + G*u(t)
     var(u(t)) = I
 @
 m=comp;
 q=zeros(ns,rows(m));
 q[1:ns,1:ns]=eye(ns);
 g=zeros(cols(m),ns);
 g[1:ns,1:ns]=(chol(seps))';
endif;

@ Pad eps with missing values @
n1=n1+tfirst-1;
if n1 .> 0;
 tmp=zeros(n1,cols(eps));
 tmp=miss(tmp,0);
 eps=tmp|eps;
endif;
n2=n2+rows(yv)-tlast;
if n2 .> 0;
 tmp=zeros(n2,cols(eps));
 tmp=miss(tmp,0);
 eps=eps|tmp;
endif;

retp(beta, seps, eps, nobs, ndf, v_bs, q, m, g);

endp;

/* ----------------------------------------- */

proc(2) = varspec(s,lag,nord,tfirst,tlast);

/* Computes VAR and covariance matrix of estimated parameters
   
   Input:
   		    s = txn matrix of series
   		    lag = number of lags
   		    nord = number of ordinates
   		    tfirst = index of first obs to use
   		    tlast = index of last obs to use

   Output:
          specmat = nord x (n*n) spectrum (normalized by 2pi)
          wvec = vector of ordinates          
*/

local step, wvec, ns, specmat, ivarmat, icomp, beta, seps, nobs, ndf, v_bs, q, m, g, i, w, ss;

step=pi/nord;
wvec=seqa(step,step,nord); 
ns=cols(s);
specmat=miss(zeros(nord,ns*ns),0);

 ivarmat=0;
 icomp=1;
 {beta, seps, nobs, ndf, v_bs, q, m, g} =varest(s,lag,tfirst,tlast,ivarmat,icomp);
 i=1; do while i <= nord;
  w=wvec[i];
  ss=spmodh(q,m,g,w);
  specmat[i,.]=(vec(ss))';
 i=i+1; endo;
 specmat=specmat/(2*pi);
 retp(specmat,wvec);
endp;

/* ----------------------------------------------------------------- */

proc(1) = spmodh(h,f,g,w);
/* Procedure for Calculating Spectral Density Matrix for a VAR of the
   form:
     x(t)=h*s(t)
     s(t)=f*s(t-1) + g*e(t)

     where var(e(t))=I

   Output:  ss = spectrum of x at frequency w.
   Note: Spectrum is not divided by 2*pi

*/
local im, z, sm, smi, ss;


 let im = 0+1i;
 z=exp(-w*im);
 sm=eye(rows(f));
 sm=sm-z*f;
 smi=inv(sm);
 ss=h*smi*g*g'*smi'*h';

retp(ss);
endp;

/* ----------------------------------------------------------------- */