
% Monte Carlo for Annual Review; 
% Quantitative Spatial Model;
% Helpman (1998) model;
% Countries and regions;

% July, 2016;

% *********************; 
% **** Choose User ****; 
% *********************;

clear all;
clc;
colormap hot;
close all;

user=2;

% User 1 : steve mac home; 
% User 2 : esteban desktop;

if user==1;
cd /Users/reddings/Dropbox/ARQSM/matlab/ARE/;
addpath /Users/reddings/Dropbox/ARQSM/matlab/ARE/functions;
end;

if user==2;
cd /Users/erossi.PRINCETON/Dropbox/ARQSM/matlab/ARE/;
addpath /Users/erossi.PRINCETON/Dropbox/ARQSM/matlab/ARE/functions;
end;


format shortG;

global alpha sigma LL LLwest LLeast F;

% ************************; 
% **** Initialization ****;
% ************************;

% Set default random number stream;
s = RandStream('mt19937ar','Seed',1);
RandStream.setGlobalStream(s);

% *************************; 
% **** Distance matrix ****;
% *************************;

% Locations on a N * N latitude and longitude grid
% Implies N * N locations
% Distance matrix is N * N 

N=30;
NN=N*N;
% Other latitude-longitude grid;
ltd=linspace(0,4,N)';
lgd=linspace(0,4,N);

% Transport costs for each point on grid;
tt=1; 
tau=zeros(N);
tau(:,:)=tt;

% Compute weighted distance using these transport costs;

dist=zeros(NN);

for z=1:NN;
    seed=false(N,N);
    seed(z)=true;
    temp=graydist(tau,seed,'quasi-euclidean');
    dist(z,:)=reshape(temp,1,NN); 
end;

% Own iceberg transport costs are one
dist(eye(size(dist))==1)=1;

% Trade costs are a power function of effective distance;
dist=dist.^0.33;

% Define east and west as two countries;
Iwest=zeros(N,N);
Ieast=zeros(N,N);
Iwest(:,1:N/2)=1;
Ieast(:,N/2+1:N)=1;
Iwest=reshape(Iwest,NN,1);
Ieast=reshape(Ieast,NN,1);

% Border friction between grid points;
bord=ones(NN,NN);
bord(:,:)=2;
bord(eye(size(bord))==1)=1;

% Border friction between countries
bordcty=ones(NN,NN);
bordcty(Iwest==1,Ieast==1)=2;
bordcty(Ieast==1,Iwest==1)=2;

% Counterfactual border friction between grid points;
cbord=ones(NN,NN);

% Counterfactual border friction between countries;
cbordcty=ones(NN,NN);

% **************************;
% **** Parameterization ****;
% **************************;

% Share of goods in consumption expenditure (1-housing share);
alpha=0.75; 
% Elasticity of substitution;
sigma=5;

% ************************************;
% **** Random productivity shocks ****;
% ************************************;

a=normrnd(0,1,NN,1);
a=exp(a);
a(Iwest==1)=a(Iwest==1)./geomean(a(Iwest==1));
a(Ieast==1)=a(Ieast==1)./geomean(a(Ieast==1));

display('Summary statistics productivities');
display('mean(a) std(a) max(a) min(a)');
[mean(a) std(a) max(a) min(a)]

% **************************;
% **** Other Parameters ****;
% **************************:

% Land area;
H=100.*ones(NN,1);
% Aggregate labor Supply;
LL=153889;          % US civilian labor force 2010 (Statistical Abstract, millions);
LLwest=(sum(Iwest)./(sum(Iwest)+sum(Ieast))).*LL;
LLeast=(sum(Ieast)./(sum(Iwest)+sum(Ieast))).*LL;
% Fixed production cost;
F=1;

% ********************************;
% **** Matrix of Fundamentals ****;
% ********************************;

fund=zeros(NN,4);
fund(:,1)=a; fund(:,2)=H; fund(:,3)=Iwest; fund(:,4)=Ieast;

% ****************************************************************************;
% **** Open Economy Solve for Endogenous Variables in Initial Equilibrium ****;
% ****************************************************************************;

display('>>>> Start Wage and Population Convergence <<<<');
[w,L,tradesh,dtradesh,converge,xtic]=solveHLwCtyOpen_E(fund,dist,bord,bordcty,NN);
display('>>>> Wage and Population System Converged <<<<');
display('>>>> Check Wage and Population Convergence (Should ==1) <<<<');
[converge]
display('>>>> Elapsed Time in Seconds <<<<');
xtic

% Price index;
[P] = Hpindex(fund,L,w,dtradesh);

% Land prices;
[r]=Hlandprice(fund,L,w);

% Real wage;
[realwage]=Hrealw(fund,L,w,tradesh);

% ************************************************************************;
% ***** Counterfactual Eliminating Border Frictions Between Countries ****;
% ************************************************************************;

display('>>>> Start Wage and Population Convergence <<<<');
[cw,cL,ctradesh,cdtradesh,cconverge,xtic]=solveHLwCtyOpen_E(fund,dist,bord,cbordcty,NN);
display('>>>> Wage and Population System Converged <<<<');
display('>>>> Check Wage and Population Convergence (Should ==1) <<<<');
[cconverge]
display('>>>> Elapsed Time in Seconds <<<<');
xtic

% Price index;
[cP] = Hpindex(fund,cL,cw,cdtradesh);

% Land prices;
[cr]=Hlandprice(fund,cL,cw);

% Real wage;
[crealwage]=Hrealw(fund,cL,cw,ctradesh);

% Welfare gains;
[welfgain]=Hwelfaregains(ctradesh,tradesh,cL,L);
display('>>>> Welfare Gains <<<<');
welfgain=round(welfgain.*(10.^4));
welfgain=welfgain./(10.^4);
unique(welfgain)

% **************************************************************************;
% ***** Counterfactual Eliminating Border Frictions Between Grid Points ****;
% **************************************************************************;

display('>>>> Start Wage and Population Convergence <<<<');
[ccw,ccL,cctradesh,ccdtradesh,ccconverge,xtic]=solveHLwCtyOpen_E(fund,dist,cbord,bordcty,NN);
display('>>>> Wage and Population System Converged <<<<');
display('>>>> Check Wage and Population Convergence (Should ==1) <<<<');
[ccconverge]
display('>>>> Elapsed Time in Seconds <<<<');
xtic

% Price index;
[ccP] = Hpindex(fund,ccL,ccw,ccdtradesh);

% Land prices;
[ccr]=Hlandprice(fund,ccL,ccw);

% Real wage;
[ccrealwage]=Hrealw(fund,ccL,ccw,cctradesh);

% Welfare gains;
[welfgain]=Hwelfaregains(cctradesh,tradesh,ccL,L);
display('>>>> Welfare Gains <<<<');
welfgain=round(welfgain.*(10.^4));
welfgain=welfgain./(10.^4);
unique(welfgain)

% ***********************************************;
% **** Three-Dimensional Initial Equilibrium ****;
% ***********************************************;

% LOG PRODUCTIVITY;
amat=reshape(log(a),N,N);

% LOG POPULATION;
Lmat=reshape(log(L),N,N);

% LOG WAGE;
wmat=reshape(log(w),N,N);

% LOG RELATIVE LAND PRICE;
rmat=reshape(log(r),N,N);

% PRICE INDEX;
Pmat=reshape(log(P),N,N);

% MULTI-PANEL FIGURE;
figure(1);
% Productivity;
C=imagesc(amat);
xlabel('Longitude','FontSize',12);
ylabel('Latitude','FontSize',12);
title('Log Productivity','FontSize',12);
set(gca,'fontsize',12);
set(gca,'YDir','normal')
set(gcf, 'PaperOrientation', 'landscape');
set(gcf, 'PaperPositionMode', 'manual');
set(gcf, 'PaperUnits', 'inches');
set(gcf, 'PaperPosition', [0.25 0.25 10.5 8]);
colorbar
print -dpdf graphs/H_cnty_initial_prod.pdf;


% MULTI-PANEL FIGURE;
figure(2);

% Population;
subplot(2,2,1);
C=imagesc(Lmat);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel A : Log Population','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
% Wages;
subplot(2,2,2);
C=imagesc(wmat);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel B : Log Wages','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
% Land prices;
subplot(2,2,3);
C=imagesc(rmat);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel C : Log Land Prices','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
% Price Index;
subplot(2,2,4);
C=imagesc(Pmat);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel D : Log Price Index','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
set(gcf, 'PaperOrientation', 'landscape');
set(gcf, 'PaperPositionMode', 'manual');
set(gcf, 'PaperUnits', 'inches');
set(gcf, 'PaperPosition', [0.25 0.25 10.5 8]);
print -dpdf graphs/H_cnty_initial.pdf;

% **************************************************************************;
% **** Three-Dimensional Eliminating Border Frictions Between Countries ****;
% **************************************************************************;

dL=cL./L; ldL=log(dL);
dw=cw./w; ldw=log(dw);
dr=cr./r; ldr=log(dr);
dP=cP./P; ldP=log(dP);

% POPULATION;
dLmat=reshape(ldL,N,N);

% PRICE INDEX;
dPmat=reshape(ldP,N,N);

% WAGE;
dwmat=reshape(ldw,N,N);

% RELATIVE LAND PRICE;
drmat=reshape(ldr,N,N);

% MULTI-PANEL FIGURE DIFFERENCES;
figure(3);
% Population;
subplot(2,2,1);
C=imagesc(dLmat);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel A : Log Relative Population','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
% Wage;
subplot(2,2,2);
C=imagesc(dwmat);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel B : Log Relative Wages','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
% Land price;
subplot(2,2,3);
C=imagesc(drmat);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel C : Log Relative Land Rents','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
% Price index;
subplot(2,2,4);
C=imagesc(dPmat);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel D : Log Relative Price Index','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
set(gcf, 'PaperOrientation', 'landscape');
set(gcf, 'PaperPositionMode', 'manual');
set(gcf, 'PaperUnits', 'inches');
set(gcf, 'PaperPosition', [0.25 0.25 10.5 8]);
print -dpdf graphs/H_cnty_c.pdf;

% MULTI-PANEL FIGURE LEVELS;
figure(4);
% Population;
subplot(2,2,1);
C=imagesc(Lmat + dLmat);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel A : Log Population','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
% Wage;
subplot(2,2,2);
C=imagesc(wmat + dwmat);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel B : Log Wages','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
% Land price;
subplot(2,2,3);
C=imagesc(rmat + drmat);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel C : Log Land Rents','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
% Price index;
subplot(2,2,4);
C=imagesc(Pmat + dPmat);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel D : Log Price Index','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
set(gcf, 'PaperOrientation', 'landscape');
set(gcf, 'PaperPositionMode', 'manual');
set(gcf, 'PaperUnits', 'inches');
set(gcf, 'PaperPosition', [0.25 0.25 10.5 8]);
print -dpdf graphs/H_cnty_c_lev.pdf;

% ****************************************************************************;
% **** Three-Dimensional Eliminating Border Frictions Between Grid Points ****;
% ****************************************************************************;

ddL=ccL./L; lddL=log(ddL);
ddw=ccw./w; lddw=log(ddw);
ddr=ccr./r; lddr=log(ddr);
ddP=ccP./P; lddP=log(ddP);

% POPULATION;
ddLmat=reshape(lddL,N,N);

% PRICE INDEX;
ddPmat=reshape(lddP,N,N);

% WAGE;
ddwmat=reshape(lddw,N,N);

% RELATIVE LAND PRICE;
ddrmat=reshape(lddr,N,N);

% MULTI-PANEL FIGURE DIFFERENCES;
figure(5);
% Population;
C=subplot(2,2,1);
imagesc(ddLmat);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel A : Log Relative Population','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
% Wage;
subplot(2,2,2);
C=imagesc(ddwmat, [-0.05 0.05]);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel B : Log Relative Wages (Truncated)','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
% Land price;
subplot(2,2,3);
C=imagesc(ddrmat);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel C : Log Relative Land Rents','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
% Price index;
subplot(2,2,4);
C=imagesc(ddPmat, [-1.35 -1.15]);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel D : Log Relative Price Index (Truncated)','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
set(gcf, 'PaperOrientation', 'landscape');
set(gcf, 'PaperPositionMode', 'manual');
set(gcf, 'PaperUnits', 'inches');
set(gcf, 'PaperPosition', [0.25 0.25 10.5 8]);
print -dpdf graphs/H_cnty_cc.pdf;


% MULTI-PANEL FIGURE LEVELS;
figure(6);
% Population;
subplot(2,2,1);
C=imagesc(Lmat + ddLmat);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel A : Log Population','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
% Wage;
subplot(2,2,2);
C=imagesc(wmat + ddwmat);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel B : Log Wages','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
% Land price;
subplot(2,2,3);
C=imagesc(rmat + ddrmat);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel C : Log Land Rents','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
% Price index;
subplot(2,2,4);
C=imagesc(Pmat + ddPmat);
xlabel('Longitude','FontSize',8);
ylabel('Latitude','FontSize',8);
title('Panel D : Log Price Index','FontSize',8);
set(gca,'fontsize',8);
set(gca,'YDir','normal')
colorbar
set(gcf, 'PaperOrientation', 'landscape');
set(gcf, 'PaperPositionMode', 'manual');
set(gcf, 'PaperUnits', 'inches');
set(gcf, 'PaperPosition', [0.25 0.25 10.5 8]);
print -dpdf graphs/H_cnty_cc_lev.pdf;