
% Simulations for Bernard, Redding, Schott (2006), "Comparative Advantage and Heterogenous Firms"; 

% Autarky equilibrium heterogeneous firm model;
% SJR, March 05, 2005;

clear;
clc;
format long;

global b1 b2 alpha Ws S s L l;

% parameters;

sigma=3.8;            % elasticity of substitution between varieties;
rho=1-1/sigma;
alpha=0.5;          % cobb-douglas expenditure share on skill-intensive good;

beta1=0.6;          % cobb-douglas exponent on skilled labor in industry 1;
beta2=0.4;          % cobb-douglas exponent on skilled labor in industry 2;
f1=0.10;               % fixed production cost in industry 1;
f2=0.10;               % fixed production cost in industry 2;
fe1=2;              % fixed entry cost in industry 1;
fe2=2;              % fixed entry cost in industry 2;
delta=0.025          % probability of death;

S=1200;             % skilled endowment home;
L=1000;             % labor endowment home;
s=1000;             % skilled endowment foreign;
l=1200;             % labor endowment foreign;

Ws=1;               % choose skilled wage for numeraire in home;
wl=1;               % choose unskilled wage for numeraire in foreign;

k=0.2;              % pareto parameters;
c=3.4; 
gamma=c-sigma+1;    
xi=c*(k^(c-gamma))/gamma;

% Starting values taken from the free trade equilibrium of the model; 

Wl=1; ws=1;
L1=640; L2=360; S1=960; S2=240; 
l1=240; l2=960; s1=360; s2=640; 
R1=1600; R2=600; r1=600; r2=1600;
R=2200; r=2200;

% *******************************************************************;
% ** Solve the system of non-linear equations for home equilibrium **;
% *******************************************************************;

start=([R R1 R2 Wl]);
start_i=start*10000;
start_i=round(start_i);

i=1;

while i<10000;

R=Wl*L+Ws*S;
R1=alpha*R;
R2=(1-alpha)*R;
Wl=((1-beta1)*R1+(1-beta2)*R2)/L;

new=([R R1 R2 Wl]);
new_i=new*10000;
new_i=round(new_i);
test=new_i-start_i;

if test==0;
    i=10000;
    disp('Home system converged')
else;
    start=(0.7*start)+(0.3*new);
    start_i=start*10000;
    start_i=round(start_i);

    R =start(1);
    R1=start(2);
    R2=start(3);
    Wl=start(4);
    
    i=i+1;
end;
end;

delete C:\Redding\Matlab\HoHetAutarky\autarky_output.out;
diary('C:\Redding\Matlab\HoHetAutarky\autarky_output.out');

disp('Solution Vector')

R
R1
R2
Ws
Wl

disp('Equilibrium Labor Allocation')
S1=beta1*R1/Ws
S2=beta2*R2/Ws
L1=(1-beta1)*R1/Wl
L2=(1-beta2)*R2/Wl
 
disp('Equilibrium productivity cutoffs')
PHI1= ((f1/delta)*(c/gamma-1)*k^c)^(1/c) * (1/fe1)^(1/c)
PHI2= ((f2/delta)*(c/gamma-1)*k^c)^(1/c) * (1/fe2)^(1/c) 
disp('Equilibrium weighted average productivity')
A1=(c/gamma)^(1/(sigma-1)) * PHI1
A2=(c/gamma)^(1/(sigma-1)) * PHI2
disp('Equilibrium product prices')
Q1=(1/rho) * (1/A1) * (Ws^beta1) * (Wl^(1-beta1))
Q2=(1/rho) * (1/A2) * (Ws^beta2) * (Wl^(1-beta2))
disp('Equilibrium firm revenues')
AR1=(A1/PHI1)^(sigma-1) * sigma * f1 * Ws^beta1 * Wl^(1-beta1)
AR2=(A2/PHI2)^(sigma-1) * sigma * f2 * Ws^beta2 * Wl^(1-beta2)
disp('Equilibrium masses of firms')
M1=R1/AR1
M2=R2/AR2
disp('Equilibrium price indices')
P1=(M1*Q1^(1-sigma))^(1/(1-sigma))
P2=(M2*Q2^(1-sigma))^(1/(1-sigma))
disp('Mass of entrants')
ME1=delta .* (PHI1./k).^c .* M1
ME2=delta .* (PHI2./k).^c .* M2
disp('Equilibrium expenditure')
E1=P1^(sigma-1) * Q1^(1-sigma) * alpha * R * M1 
E2=P2^(sigma-1) * Q2^(1-sigma) * (1-alpha) * R * M2 

% **********************************************************************;
% ** Solve the system of non-linear equations for foreign equilibrium **;
% **********************************************************************;

start=([r r1 r2 ws]);
start_i=start*10000;
start_i=round(start_i);

i=1;

while i<10000;

r=wl*l+ws*s;
r1=alpha*r;
r2=(1-alpha)*r;
ws=(beta1*r1+beta2*r2)/s;

new=([r r1 r2 ws]);
new_i=new*10000;
new_i=round(new_i);
test=new_i-start_i;

if test==0;
    i=10000;
    disp('Foreign system converged')
else;
    start=(0.7*start)+(0.3*new);
    start_i=start*10000;
    start_i=round(start_i);

    r =start(1);
    r1=start(2);
    r2=start(3);
    ws=start(4);
    
    i=i+1;
end;
end;
 
disp('Solution Vector')

r
r1
r2
ws
wl

disp('Equilibrium Labor Allocation')
s1=beta1*r1/ws
s2=beta2*r2/ws
l1=(1-beta1)*r1/wl
l2=(1-beta2)*r2/wl
 
disp('Equilibrium productivity cutoffs')
phi1= ((f1/delta)*(c/gamma-1)*k^c)^(1/c) * (1/fe1)^(1/c)
phi2= ((f2/delta)*(c/gamma-1)*k^c)^(1/c) * (1/fe2)^(1/c) 
disp('Equilibrium weighted average productivity')
a1=(c/gamma)^(1/(sigma-1)) * phi1
a2=(c/gamma)^(1/(sigma-1)) * phi2
disp('Equilibrium product prices')
q1=(1/rho) * (1/a1) * (ws^beta1) * (wl^(1-beta1))
q2=(1/rho) * (1/a2) * (ws^beta2) * (wl^(1-beta2))
disp('Equilibrium firm revenues')
ar1=(a1/phi1)^(sigma-1) * sigma * f1 * ws^beta1 * wl^(1-beta1)
ar2=(a2/phi2)^(sigma-1) * sigma * f2 * ws^beta2 * wl^(1-beta2)
disp('Equilibrium masses of firms')
m1=r1/ar1
m2=r2/ar2
disp('Equilibrium price indices')
p1=(m1*q1^(1-sigma))^(1/(1-sigma))
p2=(m2*q2^(1-sigma))^(1/(1-sigma))
disp('Mass of entrants')
me1=delta .* (phi1./k).^c .* m1
me2=delta .* (phi2./k).^c .* m2
disp('Equilibrium expenditure')
e1=p1^(sigma-1) * q1^(1-sigma) * alpha * r * m1 
e2=p2^(sigma-1) * q2^(1-sigma) * (1-alpha) * r * m2 

% ************************************;
% **** Prepare data for outputing ****;
% ************************************;

% Converges under autarky;
converge=1;
% Infinite trade costs under autarky;
tau1value=1000;
tau2value=1000;
% Average output;
% Average revenue divided by average variety prices;
AOUT1=AR1./Q1;
AOUT2=AR2./Q2;
aout1=ar1./q1;
aout2=ar2./q2;
% Welfare under autarky;
CPI=(P1.^(alpha)).*(P2.^(1-alpha));
cpi=(p1.^(alpha)).*(p2.^(1-alpha));
WELF=R./CPI;
welf=r./cpi;
% Exporting cutoff is same as domestic cutoff;
PSI1=PHI1;
PSI2=PHI2;
psi1=phi1;
psi2=phi2;
% Ratio of two productivity cutoffs is 1;
LAMBDA1=1;
LAMBDA2=1;
lambda1=1;
lambda2=1;
% No export market, productivity same as domestic market;
B1=A1;
B2=A2;
b1=a1;
b2=a2;
% No exporting, probability of exporting is zero;
X1=0;
X2=0;
x1=0;
x2=0;
% No expenditure on foreign varieties;
G1=0;
G2=0;
g1=0;
g2=0;

% **************************************************;
% **** Output the basic set of data for autarky ****;
% **************************************************;

names={'Converge' 'tau1' 'tau2' 'ME1' 'ME2' 'me1' 'me2' 'CPI' 'cpi' 'Ws' 'Wl' 'wl' 'ws' 'L1' 'L2' 'S1' 'S2' 'l1' 'l2' 's1' 's2' 'Q1' 'Q2' 'q1' 'q2' 'P1' 'P2' 'p1' 'p2' 'E1' 'E2' 'e1' 'e2' 'G1' 'G2' 'g1' 'g2' 'R1' 'R2' 'r1' 'r2' 'R' 'r' 'M1' 'M2' 'm1' 'm2' 'LAMBDA1' 'LAMBDA2' 'lambda1' 'lambda2' 'PHI1' 'PHI2' 'phi1' 'phi2' 'PSI1' 'PSI2' 'psi1' 'psi2' 'A1' 'A2' 'a1' 'a2' 'B1' 'B2' 'b1' 'b2' 'X1' 'X2' 'x1' 'x2' 'AR1' 'AR2' 'ar1' 'ar2' 'AOUT1' 'AOUT2' 'aout1' 'aout2'};
names=names';
data=vertcat(aout1,aout2);
data=vertcat(AOUT2,data);
data=vertcat(AOUT1,data);
data=vertcat(ar2,data);
data=vertcat(ar1,data);
data=vertcat(AR2,data);
data=vertcat(AR1,data);
data=vertcat(x2,data);
data=vertcat(x1,data);
data=vertcat(X2,data);
data=vertcat(X1,data);
data=vertcat(b2,data);
data=vertcat(b1,data);
data=vertcat(B2,data);
data=vertcat(B1,data);
data=vertcat(a2,data);
data=vertcat(a1,data);
data=vertcat(A2,data);
data=vertcat(A1,data);
data=vertcat(psi2,data);
data=vertcat(psi1,data);
data=vertcat(PSI2,data);
data=vertcat(PSI1,data);
data=vertcat(phi2,data);
data=vertcat(phi1,data);
data=vertcat(PHI2,data);
data=vertcat(PHI1,data);
data=vertcat(lambda2,data);
data=vertcat(lambda1,data);
data=vertcat(LAMBDA2,data);
data=vertcat(LAMBDA1,data);
data=vertcat(m2,data);
data=vertcat(m1,data);
data=vertcat(M2,data);
data=vertcat(M1,data);
data=vertcat(r,data);
data=vertcat(R,data);
data=vertcat(r2,data);
data=vertcat(r1,data);
data=vertcat(R2,data);
data=vertcat(R1,data);
data=vertcat(g2,data);
data=vertcat(g1,data);
data=vertcat(G2,data);
data=vertcat(G1,data);
data=vertcat(e2,data);
data=vertcat(e1,data);
data=vertcat(E2,data);
data=vertcat(E1,data);
data=vertcat(p2,data);
data=vertcat(p1,data);
data=vertcat(P2,data);
data=vertcat(P1,data);
data=vertcat(q2,data);
data=vertcat(q1,data);
data=vertcat(Q2,data);
data=vertcat(Q1,data);
data=vertcat(s2,data);
data=vertcat(s1,data);
data=vertcat(l2,data);
data=vertcat(l1,data);
data=vertcat(S2,data);
data=vertcat(S1,data);
data=vertcat(L2,data);
data=vertcat(L1,data);
data=vertcat(ws,data);
data=vertcat(wl,data);
data=vertcat(Wl,data);
data=vertcat(Ws,data);
data=vertcat(cpi,data);
data=vertcat(CPI,data);
data=vertcat(me2,data);
data=vertcat(me1,data);
data=vertcat(ME2,data);
data=vertcat(ME1,data);
data=vertcat(tau2value,data);
data=vertcat(tau1value,data);
data=vertcat(converge,data);
delete C:\Redding\Matlab\HoHetAutarky\autarkysims.csv;
csvwrite('C:\Redding\Matlab\HoHetAutarky\autarkysims.csv',data);
delete C:\Redding\Matlab\HoHetAutarky\autarkynames.out;
diary('C:\Redding\Matlab\HoHetAutarky\autarkynames.out');
names
diary('off');

autstore_n=data(11:75,1);
csvwrite('C:\Redding\Matlab\HoHetAutarky\autstore_n.csv',autstore_n);

