
% Code to generate data on expenditure shares and prices;
% using random demand and marginal cost shocks;
% SJR, May 2015;
% May have trouble printing figures, depending on your system. Can try export_fig as well.

clear;
clc;
clf;
format long;

TERMINAL = '/mnt/data0/work/MPF_FINAL/';
code_dir = [TERMINAL,'codes/Appendix/'];
addpath(code_dir)

outpath = [TERMINAL,'results'];
cd(outpath);


global sigmaU sigmaF delta N T U O ppUddmat ssUddmat ppFdd ssFdd iiFdd;
format bank;

% *************************;
% **** INITIALIZATIONS ****;
% *************************;

% Set default random number stream;
s = RandStream('mt19937ar','Seed',1);
RandStream.setGlobalStream(s);

% Reset random number stream so same random numbers;
reset(s);

% **********************;
% **** REPLICATIONS ****;
% **********************;

%%%% True parameters;
sigmaU=7;
sigmaF=4;
delta=0.15;

RR=50;
esigmaUmat=zeros(RR,1);
edeltamat=zeros(RR,1);
esigmaFmat=zeros(RR,1);
edeltamat=zeros(RR,1);
esigmaFmat=zeros(RR,1);

tic
for r=1:1:RR;

display('>>>> REPLICATION <<<<');
display(r);
toc
    
% *********************************;
% **** Data Generating Process ****;
% *********************************;

% Assume heteroscedasticity across UPCs and firms;

% Levels are:
% First Dimension : Firms : 100 ;
% Second Dimension : Time : 200 quarters
% Third Dimension : UPCs : 10 per firm;
firm=1:1:500; firm=firm';
year=1:1:200; year=year';
upc=1:1:10; upc=upc';

N=size(firm,1);
T=size(year,1);
U=size(upc,1);
O=N*T*U;
indfirm=unique(firm);
indyear=unique(year);
indupc=unique(upc);
indobs=1:1:O;

%%%%% Data matrix is here;
%%%%% First dimension (rows) firm;
%%%%% Second dimension (cols) time;
%%%%% Third dimension upcs;

%%%% Product quality;
lphi_u=zeros(N,T,U);
for f=1:1:N;
for u=1:1:U;
vv=1+rand(1,1);
lphi_u(f,:,u)=normrnd(0,vv,1,T);
end;
end;
phi_u=exp(lphi_u); 
geophi_u=geomean(phi_u,3); 
geophi_u=repmat(geophi_u,[[1,1],U]);
phi_u=phi_u./geophi_u;
lphi_u=log(phi_u);

%%%% Firm quality;
lphi_f=zeros(N,T);
for n=1:1:N;
vv=1+rand(1,1);
lphi_f(n,:)=normrnd(0,vv,1,T);
end;
phi_f=exp(lphi_f); 
geophi_f=geomean(phi_f); 
geophi_f=repmat(geophi_f,[N,1]);
phi_f=phi_f./geophi_f;
lphi_f=log(phi_f);

%%%% Marginal cost;
laa_u=zeros(N,T,U);
for f=1:1:N;
for u=1:1:U;
vv=1+rand(1,1);
laa_u(f,:,u)=normrnd(0,vv,1,T); 
end;
end;
aa_u=exp(laa_u); 
geoaa_u=geomean(aa_u,3); 
geoaa_u=repmat(geoaa_u,[[1,1],U]);
aa_u=aa_u./geoaa_u;
laa_u=log(aa_u);

% ********************************************;
% **** GENERATE INITAL EXPENDITURE SHARES ****;
% **** UNIFORM DISTRIBUTION               ****;
% ********************************************;

R_g=100;

S_fi=zeros(N,T);
S_ui=zeros(N,T,U);

S_fi(:,:)=(1./N);
S_ui(:,:,:)=(1./U);

P_fi=zeros(N,T);
P_ui=zeros(N,T,U);

P_fi(:,:)=1;
P_ui(:,:,:)=1;

% ************************************************;
% **** SOLVE SYSTEM OF SIMULTANEOUS EQUATIONS ****;
% **** FOR ENDOGENOUS VARIABLES               ****;
% ************************************************;

x=1;
while x<=100000;

R_f=R_g.*S_fi;
R_u=repmat(R_f,[[1,1],U]).*S_ui;
[epsilon]=epsfirm(S_fi,S_ui);
[mu]=markup(epsilon);
P_u=(repmat(mu,[[1,1],U]).^(1./(1+delta))).*((1+delta).^(1./(1+delta))).*(aa_u.^(1./(1+delta))).*(R_u.^(delta./(1+delta)));
[P_f]=pfirm(P_u,phi_u);
[S_u]=expshare(P_u,phi_u,repmat(P_f,[[1,1],U]),sigmaU);
[P_g]=pgroup(P_f,phi_f);
[S_f]=expshare(P_f,phi_f,repmat(P_g,[N,1]),sigmaF);

test_f=S_fi-S_f; test_f=abs(test_f); test_f=round(test_f.*1000); test_f=reshape(test_f,N.*T,1);
test_u=S_ui-S_u; test_u=abs(test_u); test_u=round(test_u.*1000); test_u=reshape(test_u,N.*T.*U,1);

mtest_f=max(test_f); mtest_u=max(test_u);
mtest_f=mtest_f.*1000000; mtest_u=mtest_u.*1000000;
[x;mtest_f;mtest_u];

% Convergence tolerance;
if (mtest_f==0) & (mtest_u==0) & (x>=1000);
    display('>>>> System Converged <<<<');
    x=1000000;
end;

% Adjust expenditure shares
S_fi=(0.75.*S_fi)+(0.25.*((S_f./S_fi).*S_fi)); 
S_ui=(0.75.*S_ui)+(0.25.*((S_u./S_ui).*S_ui)); 

x=x+1;
end;

% ***********************;
% **** Observed data ****;
% ***********************;

ppU=P_u;
ssU=S_u;

% ************************;
% **** Check Solution ****;
% ************************;

% Check Shares sum to one;
test_f=sum(S_f,1); test_u=sum(S_u,3);
test_f=reshape(test_f,numel(test_f),1); 
test_u=reshape(test_u,numel(test_u),1); 

display('>>>> Check Shares sum to one <<<<');
display('>>>> Test Variables Should all Equal One <<<<');
[max(test_f) min(test_f)]
[max(test_u) min(test_u)]

% Check recover correct unobservables from observed data;
% using correct parameters;
gppU=geomean(ppU,3);
gppU=repmat(gppU,[[1,1],U]);
gssU=geomean(ssU,3);
gssU=repmat(gssU,[[1,1],U]);
estphi_u=(ppU./gppU).*((ssU./gssU).^(1./(sigmaU-1)));

% Check geometric mean estimated product quality;
display('>>>> Check geometric mean of estimated product quality is one <<<<');
display('>>>> Test Variable Should Equal One <<<<');
mean(mean(geomean(estphi_u,3)))

display('>>>> Check correlation of true and estimated product quality <<<<');
display('>>>> Matrix of correlation coefficients should be one <<<<');
test1=reshape(phi_u,N.*T.*U,1);
test2=reshape(estphi_u,N.*T.*U,1);
corrcoef(test1,test2)

% Check geometric mean estimated marginal cost;
estaa_u=(P_u.^(1+delta))./(repmat(mu,[[1,1],U]).*(1+delta).*(R_u.^delta));
display('>>>> Check geometric mean of estimated cost shock is one <<<<');
display('>>>> Test Variable Should Equal One <<<<');
mean(mean(geomean(estaa_u,3)))

display('>>>> Check correlation of true and estimated cost shock <<<<');
display('>>>> Matrix of correlation coefficients should be one <<<<');
test3=reshape(aa_u,N.*T.*U,1);
test4=reshape(estaa_u,N.*T.*U,1);
corrcoef(test3,test4)

% figure(1);
% scatter(log(test1),log(test2));
% title('Check Recovered Phi_i');
% xlabel('True assumed phi_u');
% ylabel('Recovered phi_u for correct parameter');
% 
% print -dpdf figures/check_phi_u.pdf;

% **************************************;
% **** Construct Double Differences ****;
% **************************************;

%ppUd=log(ppU(:,:,:));
ppUd=log(ppU(:,:,2:U))-log(ppU(:,:,1:U-1));
ppUdd=ppUd(:,2:T,:)-ppUd(:,1:T-1,:);

%ssUd=log(ssU(:,:,:));
ssUd=log(ssU(:,:,2:U))-log(ssU(:,:,1:U-1));
ssUdd=ssUd(:,2:T,:)-ssUd(:,1:T-1,:);

phiUd=log(phi_u(:,:,2:U))-log(phi_u(:,:,1:U-1));
phiUdd=phiUd(:,2:T,:)-phiUd(:,1:T-1,:);

aaUd=log(aa_u(:,:,2:U))-log(aa_u(:,:,1:U-1));
aaUdd=aaUd(:,2:T,:)-aaUd(:,1:T-1,:);

estphiUd=log(estphi_u(:,:,2:U))-log(estphi_u(:,:,1:U-1));
estphiUdd=estphiUd(:,2:T,:)-estphiUd(:,1:T-1,:);

estaaUd=log(estaa_u(:,:,2:U))-log(estaa_u(:,:,1:U-1));
estaaUdd=estaaUd(:,2:T,:)-estaaUd(:,1:T-1,:);

% **************************************************************;
% **** Check Moment Conditions for Correct Parameter Values ****;
% **************************************************************;

phivec=reshape(phiUdd,numel(phiUdd),1);
aavec=reshape(aaUdd,numel(aaUdd),1);
ppvec=reshape(ppUdd,numel(ppUdd),1);
ssvec=reshape(ssUdd,numel(ssUdd),1);
estphivec=reshape(estphiUdd,numel(estphiUdd),1);
estaavec=reshape(estaaUdd,numel(estaaUdd),1);

test1=ppUdd-((1./(1-sigmaU)).*ssUdd);
test2=((1+delta).*ppUdd)-(delta.*ssUdd);
test1=reshape(test1,numel(test1),1);
test2=reshape(test2,numel(test2),1);
% [test1 phivec estphivec] 
% [test2 aavec estaavec] 

% *******************************************************;
% **** Reshape data for UPC by UPC moment conditions ****;
% *******************************************************;

% Rows are firms and UPCs;
% Columns are years;

obs=N.*(U-1);
ppUddmat=zeros(obs,T-1);
ssUddmat=zeros(obs,T-1);
index=1;

for f=1:1:N;
display(f);
for u=1:1:U-1;
ppUddmat(index,:)=ppUdd(f,:,u);
ssUddmat(index,:)=ssUdd(f,:,u);
index=index+1;
end;
end;

% covarphiaa=(1./numel(aavec)).*(phivec'*aavec);
% 
% varpp=(1./numel(ppvec)).*(ppvec'*ppvec);
% varss=(1./numel(ssvec)).*(ssvec'*ssvec);
% covarppss=(1./numel(ssvec)).*(ssvec'*ppvec);
% 
% test=((1+delta).*varpp)-((delta+((1+delta)./(1-sigmaU))).*covarppss)+((delta./(1-sigmaU)).*varss);
% 
% display('[covarphiaa; test]');
% [covarphiaa; test]

% ******************************;
% **** OPTIMIZATION SIGMA U ****;
% ******************************;

% Theta=[esigmaU; edelta];
Theta0=[2;0.7];
LB=[1.05;0.1];
UB=[50;0.9];

% FMINCON;
options = optimset('display','iter','TolFun',eps,'TolX',eps,'Algorithm','interior-point');
[Theta,ll,EXITFLAG,OUTPUT,LAMBDA] = ...
   fmincon('estimsigmaU',Theta0,[],[],[],[],LB,UB,[],options);

esigmaU=Theta(1);
edelta=Theta(2);

esigmaUmat(r)=esigmaU;
edeltamat(r)=edelta;

% **********************************;
% **** CONSTRUCT FIRM VARIABLES ****;
% **********************************;

gppU=geomean(ppU,3);
gppU=repmat(gppU,[[1,1],U]);
gssU=geomean(ssU,3);
gssU=repmat(gssU,[[1,1],U]);
% lestphi_u=log(ppU./gppU)-(log(ssU./gssU)./(1-esigmaU));
% estphi_u=exp(lestphi_u);
estphi_u=(ppU./gppU).*((ssU./gssU).^(1./(esigmaU-1)));

test1=reshape(phi_u,N.*T.*U,1);
test2=reshape(estphi_u,N.*T.*U,1);

ppF=(ppU./estphi_u).^(1-esigmaU);
ppF=sum(ppF,3);
ppF=ppF.^(1./(1-esigmaU));

ssF=S_f;

ppFd=log(ppF(2:N,:))-log(ppF(1:N-1,:));
ppFdd=ppFd(:,2:T)-ppFd(:,1:T-1);

ssFd=log(ssF(2:N,:))-log(ssF(1:N-1,:));
ssFdd=ssFd(:,2:T)-ssFd(:,1:T-1);

% ********************;
% **** INSTRUMENT ****;
% ********************;

geoS_u=geomean(S_u,3);
geoS_u=repmat(geoS_u,[[1,1],U]);
ii=S_u./geoS_u;
ii=sum(ii,3);
ii=log(ii);

iiFd=ii(2:N,:)-ii(1:N-1,:);
iiFdd=iiFd(:,2:T)-iiFd(:,1:T-1);

% ******************************;
% **** OPTIMIZATION SIGMA F ****;
% ******************************;

Theta0=1.5;
LB=1.05;
UB=50;

% FMINCON;
options = optimset('display','iter','TolFun',eps,'TolX',eps,'Algorithm','interior-point');
[esigmaF,ll,EXITFLAG,OUTPUT,LAMBDA] = ...
   fmincon('estimsigmaF',Theta0,[],[],[],[],LB,UB,[],options);

esigmaFmat(r)=esigmaF;

end;

% *****************************;
% **** Monte Carlo Results ****;
% *****************************;

figure(1);
hist(esigmaUmat,10);
xlabel('Estimated Sigma U');
ylabel('Frequency');
hold on;
line([sigmaU sigmaU],[0 15],'Color','red','LineStyle','-','LineWidth',1.5);
axis([sigmaU-3 sigmaU+3 0 15]);
hold off;

print -dpdf figures/FigS4_montecarlosigU.pdf;

figure(2);
hist(edeltamat,10);
xlabel('Estimated Delta');
ylabel('Frequency');
hold on;
line([delta delta],[0 15],'Color','red','LineStyle','-','LineWidth',1.5);
axis([delta-0.15 delta+0.15 0 15]);
hold off;

print -dpdf figures/FigS5_montecarlodelta.pdf;

figure(3);
hist(esigmaFmat,10);
xlabel('Estimated Sigma F');
ylabel('Frequency');
hold on;
line([sigmaF sigmaF],[0 15],'Color','red','LineStyle','-','LineWidth',1.5);
axis([sigmaF-3 sigmaF+3 0 15]);
hold off;

print -dpdf figures/FigS6_montecarlosigF.pdf;

% ***************************************;
% **** Summarize Parameter Estimates ****;
% ***************************************;

format short;
diary('tables/S1_MonteCarloResults.log')
display('>>>> Mean Sigma U <<<<');
mean(esigmaUmat)
display('>>>> Standard Deviation Sigma U <<<<');
std(esigmaUmat)
display('>>>> Mean Sigma F <<<<');
mean(esigmaFmat)
display('>>>> Standard Deviation Sigma F <<<<');
std(esigmaFmat)
display('>>>> Mean Delta <<<<');
mean(edeltamat)
display('>>>> Standard Deviation Delta <<<<');
std(edeltamat)
diary off

% **********************;
% **** GRID SIGMA U ****;
% **********************;
% 
% % REGULAR SPACED GRID;
% esigmaUgrid=[1.2;2;3;4;5;6;7;8;9;10];
% edeltagrid=[0.1;0.2;0.3;0.4;0.5;0.6;0.7;0.8;0.9;1];
% 
% [esigmamat,edeltamat]=meshgrid(esigmaUgrid,edeltagrid);
% esigmavec=reshape(esigmamat,numel(esigmamat),1);
% edeltavec=reshape(edeltamat,numel(edeltamat),1);
% gmmobjU=zeros(size(esigmavec));
% 
% for z=1:1:size(esigmavec,1);
% gmmobjU(z)=estimsigmaU([esigmavec(z);edeltavec(z)]);
% end;
% 
% mingmmobjU=min(gmmobjU);
% I=(mingmmobjU==gmmobjU);
% display('[True Sigma; Estimated Sigma]');
% [sigmaU; esigmavec(I)]
% display('[True Delta; Estimated Delta]');
% [delta; edeltavec(I)]
% 

% 

exit