%% SETUP_SRV02_EXP08_SIP
%
% Sets the necessary parameters to run the SRV02 Experiment #8: Single
% Inverted Pendulum laboratory using the "s_sip", "q_sip", and "q_sesip" 
% Simulink diagrams.
% 
% Copyright (C) 2010 Quanser Consulting Inc.
%
clear all;
%
%% SRV02 Configuration
% External Gear Configuration: set to 'HIGH' or 'LOW'
EXT_GEAR_CONFIG = 'HIGH';
% Encoder Type: set to 'E' or 'EHR'
ENCODER_TYPE = 'E';
% Is SRV02 equipped with Tachometer? (i.e. option T): set to 'YES' or 'NO'
TACH_OPTION = 'YES';
% Type of Load: set to 'NONE', 'DISC', or 'BAR'
LOAD_TYPE = 'NONE';
% Amplifier Gain used: set to 1, 3, or 5
K_AMP = 1;
% Amplifier Type: set to 'UPM_1503' or 'UPM_2405' or
% 'Q3' or 'VoltPaq'
AMP_TYPE = 'VoltPaq';
% Digital-to-Analog Maximum Voltage (V)
VMAX_DAC = 10;
%
%% Single-Pendulum Configuration
% ROTPEN Option: 'ROTPEN' or 'ROTPEN-E'
ROTPEN_OPTION = 'ROTPEN-E';
% Length of pendulum.
PEND_TYPE = 'MEDIUM_12IN';
% Define rotary arm attached to SRV02 load gear.
SRV02_ARM = 'ROTARY_ARM';
%
% Safety watchdog on the SRV02 arm angle: ON = 1, OFF = 0 
THETA_LIM_ENABLE = 1;       % safety watchdog turned ON
% THETA_LIM_ENABLE = 0;      % safety watchdog turned OFF
% Safety Limits on the SRV02 arm angle (deg)
THETA_MAX = 60;            % pendulum angle maximum safety position (deg)
THETA_MIN = - THETA_MAX;   % pendulum angle minimum safety position (deg)
% Safety watchdog on pendulum 1 angle: ON = 1, OFF = 0 
 ALPHA_LIM_ENABLE = 1;       % safety watchdog turned ON
 %ALPHA_LIM_ENABLE = 0;      % safety watchdog turned OFF
% Safety Limits on the pendulum 1 angle (deg)
%global ALPHA_MAX ALPHA_MIN
ALPHA_MAX = 45;            % pendulum angle maximum safety position (deg)
ALPHA_MIN = - ALPHA_MAX;   % pendulum angle minimum safety position (deg)
%
%
%% Lab Configuration
% Type of controller: set it to 'AUTO', 'MANUAL'
% CONTROL_TYPE = 'AUTO';
 CONTROL_TYPE = 'MANUAL';
%
%% Control specifications
% SRV02 Position Control specifications
% Peak time (s)
tp_srv02 = 0.15;
% Percentage overshoot (%)
PO_srv02 = 5.0;
%
%% System Parameters
% Sets model variables according to the user-defined SRV02 configuration
[ Rm, kt, km, Kg, eta_g, Beq, Jm, Jeq, eta_m, K_POT, K_TACH, K_ENC, VMAX_AMP, IMAX_AMP ] = config_srv02( EXT_GEAR_CONFIG, ENCODER_TYPE, TACH_OPTION, AMP_TYPE, LOAD_TYPE );
% Sets model variables according to the user-defined pendulum configuration
[ g, m_arm, r, l_arm, J_arm_cm, Barm, RtpnOp, RtpnOff, K_POT_PEN ] = config_sp( SRV02_ARM, ROTPEN_OPTION );
[ g, mp, Lp, lp, Jp_cm, Bp, RtpnOp, RtpnOff, K_POT_PEN ] = config_sp( PEND_TYPE, ROTPEN_OPTION );
%
% Moment of inertia acting seen from arm pivot.
Jarm = J_arm_cm + m_arm*l_arm^2;
% Moment of inertia acting seen from pendulum pivot.
Jp = Jp_cm + mp*lp^2;
% Equivalent moment of inertia at motor, i.e. with rotary arm load.
Jm = Jm + Jarm;
% Set Open-loop state-space model of rotary single inverted pendulum
SRV02_SIP_ABCD_eqns;
% Initial state for simulation
X0 = zeros(1,4);
% Sampling interval (s)
h = 1e-3;
%
%% Filter Parameters
% SRV02 High-pass filter in PD control used to compute velocity
if strcmp(ROTPEN_OPTION,'ROTPEN')
    % Cutoff frequency (rad/s)
    wcf_1 = 2 * pi * 25.0;
    % Damping ratio
    zetaf_1 = 0.9;
else
    % Cutoff frequency (rad/s)
    wcf_1 = 2 * pi * 50.0;
    % Damping ratio
    zetaf_1 = 0.9;
end
% Pendulum High-pass filter in PD control used to compute velocity
% Cutoff frequency (rad/s)
wcf_2 = 2 * pi * 8.5;
% Damping ratio
zetaf_2 = 0.9;
%
%% Calculate Control Parameters
if strcmp ( CONTROL_TYPE , 'MANUAL' )
    % Load model parameters based on SRV02 configuration.
    [ K, tau ] = d_model_param(Rm, kt, km, Kg, eta_g, Beq, Jeq, eta_m, AMP_TYPE); 
    disp ('');
    [Q,R,Er] = calculate_qr(AMP_TYPE,ROTPEN_OPTION);
%    Er = input('Please enter your estimated reference energy for swing up control: ');
    k = zeros(1,4);    
elseif strcmp ( CONTROL_TYPE , 'AUTO' )
    % Load model parameters based on SRV02 configuration.
    [ K, tau ] = d_model_param(Rm, kt, km, Kg, eta_g, Beq, Jeq, eta_m, AMP_TYPE);
    [Q,R,Er] = calculate_qr(AMP_TYPE,ROTPEN_OPTION);
    [k, S, E] = lqr(A,B,Q,R);    
end
% Gravitational acceleration (m/s^2)
g = 9.81;
% Maximum acceleration of pivot (m/s^2)
a_max = eta_m*eta_g*Kg*kt*IMAX_AMP/Jm;
% Experimentally determined maximum acceleration when moving 0.1 m (g)
u_max = a_max/g;
% Maximum acceleration factor
n = u_max / g;
% Mode-Switching Parameters
% pendulum angular range (rad)
CATCH_ALPHA_UP_LIM = 1.5 * pi / 180;
% pendulum angular velocity range (rad/s)
CATCH_ALPHA_DOT_LIM = VMAX_AMP/k(4);
%
%% Display
disp( ' ' );
disp( 'SRV02 model parameters: ' );
disp( [ '   K   = ' num2str( K, 3 ) ' rad/s/V' ] );
disp( [ '   tau = ' num2str( tau, 3 ) ' s' ] );
disp( ' ' );
disp( 'SRV02+SIP balance control gains: ' );
disp( [ '   k(1) = ' num2str( k(1) ) ' N.m/rad' ] );
disp( [ '   k(2) = ' num2str( k(2) ) ' N.m/rad' ] );
disp( [ '   k(3) = ' num2str( k(3) ) ' N.m/(rad/s)' ] );
disp( [ '   k(4) = ' num2str( k(4) ) ' N.m/(rad/s)' ] );