

clear all; close all;

% Two-period model of sorting across locations with amenities


%% SETUP

% locations
z  =  0:0.01:2 ;
Nz = length(z) ;

% initial asset endowments
a_low  = 0 ;
a_med  = 1 ;
a_high = 2 ;

% first period wages
w      = 1 ;
w_low  = w; 
w_med  = w; 
w_high = w; 

% Second period wages
w_new      = 0.5 ;
w_low_new  = w_new ;
w_high_new = w_new ;
w_med_new  = w_new ;

% amenities
A = 2 ; 
Amenities = A * z ;

% disposable income
y_low  = w_low + a_low; 
y_high = w_high + a_high; 
y_med  = w_med + a_med; 

y_low_new  = w_low_new + a_low;
y_high_new = w_high_new + a_high;
y_med_new  = w_med_new + a_med ;

% possible smoothing parameter to deal with zeros. set to 0.
eps = 0 ; 

theta = 3;      % idiosyncratic sorting elasticity
alpha = 1.1; % how rent increases with z
q_z = z.^alpha; % rent (note that this can be assumed by specifying a particular underlying land supply correlated with z)


%% LOCATION CHOICE PROBABILITIES: IID PREFERENCE SHOCKS CASE

    % first period
    
pi_low  = max((exp(Amenities).*(y_low - q_z)).^theta,eps); 
pi_low  = pi_low ./ sum(pi_low);

pi_med  = max((exp(Amenities).*(y_med - q_z)).^theta,eps); 
pi_med  = pi_med ./ sum(pi_med);

pi_high = max((exp(Amenities).*(y_high - q_z)).^theta,eps);
pi_high = pi_high ./ sum(pi_high);

    % second period

pi_low_new = max((exp(Amenities).*(y_low_new - q_z)).^theta,eps); 
pi_low_new = pi_low_new ./ sum(pi_low_new);

pi_med_new = max((exp(Amenities).*(y_med_new - q_z)).^theta,eps); 
pi_med_new = pi_med_new ./ sum(pi_med_new);    

pi_high_new = max((exp(Amenities).*(y_high_new - q_z)).^theta,eps); 
pi_high_new = pi_high_new ./ sum(pi_high_new);


%% CONDITIONAL MEANS AND DiD: IID PREFERENCE SHOCKS CASE

PiMatLow = pi_low' * pi_low_new ;
PiMatHigh = pi_high' * pi_high_new ;
PiMatMed = pi_med' * pi_med_new ;

dz = ones(Nz,1) * z - z' * ones(1,Nz) ;

dzLowCond  = sum( PiMatLow  .* dz , 2 ) ./ sum( PiMatLow  , 2 ) ;
dzLowCond(isnan(dzLowCond)) = 0 ;
dzHighCond = sum( PiMatHigh .* dz , 2 ) ./ sum( PiMatHigh , 2 ) ;
dzHighCond(isnan(dzHighCond)) = 0 ;
dzMedCond = sum( PiMatMed .* dz , 2 ) ./ sum( PiMatMed , 2 ) ;
dzMedCond(isnan(dzMedCond)) = 0 ;

InitLocLow  = sum( PiMatLow  , 2 ) ;
InitLocHigh = sum( PiMatHigh , 2 ) ;
InitLocMed = sum( PiMatMed , 2 ) ;

% Unconditional DiD
dzLow  = sum( PiMatLow(:)  .* dz(:) ) / sum( PiMatLow(:) ) ;
dzHigh = sum( PiMatHigh(:) .* dz(:) ) / sum( PiMatHigh(:) ) ;
dzMed = sum( PiMatMed(:) .* dz(:) ) / sum( PiMatMed(:) ) ;
ddzMean    = dzLow - dzHigh ;
ddzMeanMed = dzMed - dzHigh ;


%% PLOT FIGURE 10(b) 

shocks = 'IID' ;
run FiguresAmenities.m


%% CONDITIONAL MEANS AND DiD: PERSISTENT PREFERENCE SHOCKS CASE
 
rng(25) ;

% flow payoff in first period
u_low        = max((exp(Amenities).*(y_low - q_z)),eps) ;
u_high       = max((exp(Amenities).*(y_high - q_z)),eps) ;
u_med        = max((exp(Amenities).*(y_med - q_z)),eps) ;

% flow payoff in second period
u_low_new    = max((exp(Amenities).*(y_low_new - q_z)),eps) ;
u_high_new   = max((exp(Amenities).*(y_high_new - q_z)),eps) ;
u_med_new    = max((exp(Amenities).*(y_med_new - q_z)),eps) ;

% payoff in first and second period (now 0 and 1) raied to the power theta
uLow0Theta  = u_low.^theta ;
uHigh0Theta = u_high.^theta ;
uMed0Theta  = u_med.^theta ;
uLow1Theta  = u_low_new.^theta ;
uHigh1Theta = u_high_new.^theta ;
uMed1Theta  = u_med_new.^theta ;

% sum of payoffs net of a given payoff, useful for conditional choice
% probabilites
PhiLow0  = sum( uLow0Theta ) - uLow0Theta ;
PhiLow1  = sum( uLow1Theta ) - uLow1Theta ;
PhiHigh0 = sum( uHigh0Theta ) - uHigh0Theta ;
PhiHigh1 = sum( uHigh1Theta ) - uHigh1Theta ;
PhiMed0  = sum( uMed0Theta ) - uMed0Theta ;
PhiMed1  = sum( uMed1Theta ) - uMed1Theta ;    
    
% time 0 will be rows, and time 1 will be columns

% initialize transition probabilities
PiMatLow = zeros(Nz,Nz) ;
PiMatHigh = zeros(Nz,Nz) ;
PiMedHigh = zeros(Nz,Nz) ;
NI = 1000 ;
 
% grid for integration
phig = linspace(0,pi/2,NI)'; % rows are Y grid points
dphig = [ phig(2:end) - phig(1:(end-1)) ; phig(end) - phig(end-1) ] ;
 
% loop over all possible location pairs to compute bilateral migration
% probabilities

% use formula in

% Balboni, C., Bryan, G., Morten, M., & Siddiqi, B. (2020).
% Transportation, Gentrification, and Urban Mobility: The Inequality Effects of Place-Based Policies.

% with continuuous space, formula rewrites as integral over trigonometric
% functions
% email adrienbilal@fas.harvard.edu for details

for z0=1:Nz
    for z1=1:Nz
                
        % Low types
        
        R = ( uLow1Theta(z1) * uLow0Theta(z0) ) / ( uLow0Theta(z1) * uLow1Theta(z0) ) ;
        
        if z0 == z1
            
            PiMatLow(z0,z1) =  uLow0Theta(z0) * uLow1Theta(z0) / sum( max( uLow0Theta(z0) * uLow1Theta , uLow1Theta(z0) * uLow0Theta ) ) ;
            
        elseif z0 ~= z1
            
            if isnan(R) == 0 && R > 1
                                
                phiA = atan( uLow0Theta(z1) / uLow1Theta(z1) ) ;
                phiB = atan( uLow0Theta(z0) / uLow1Theta(z0) ) ;
                
                ls = ( logspace(0,1,NI)' - 1 ) / 9 ;
                
                phi1 = phiA + ( ( phiB + phiA ) / 2 - phiA ) ...
                     * ls ;
                 
                phi2 = ( phiB + phiA ) / 2 + ( phiB - ( phiB + phiA ) / 2 ) ...
                     * ( 1-fliplr(ls')' ) ;
                
                phig = [ phi1 ; phi2 ] ;
                
                dphig = [ phig(2:end) - phig(1:(end-1)) ; phig(end) - phig(end-1) ] ;

                InnerMax = max( sin(phig) * uLow1Theta  , ...
                                cos(phig) * uLow0Theta ) ;
                Integrand = 1./ sum(InnerMax,2).^2 ;
                Integral = Integrand' * dphig ;

                PiMatLow(z0,z1) = Integral * uLow0Theta(z0) * uLow1Theta(z1) ;
                
            end
            
        end
            
        % High types
        
        R = ( uHigh1Theta(z1) * uHigh0Theta(z0) ) / ( uHigh0Theta(z1) * uHigh1Theta(z0) ) ;
        
        if z0 == z1
            
            PiMatHigh(z0,z1) =  uHigh0Theta(z0) * uHigh1Theta(z0) / sum( max( uHigh0Theta(z0) * uHigh1Theta , uHigh1Theta(z0) * uHigh0Theta ) ) ;
            
        elseif z0 ~= z1
            
            if isnan(R) == 0 && R > 1

                                
                phiA = atan( uHigh0Theta(z1) / uHigh1Theta(z1) ) ;
                phiB = atan( uHigh0Theta(z0) / uHigh1Theta(z0) ) ;
                
                ls = ( logspace(0,1,NI)' - 1 ) / 9 ;
                
                phi1 = phiA + ( ( phiB + phiA ) / 2 - phiA ) ...
                     * ls ;
                 
                phi2 = ( phiB + phiA ) / 2 + ( phiB - ( phiB + phiA ) / 2 ) ...
                     * ( 1-fliplr(ls')' ) ;
                
                phig = [ phi1 ; phi2 ] ;
                
                dphig = [ phig(2:end) - phig(1:(end-1)) ; phig(end) - phig(end-1) ] ;

                InnerMax = max( sin(phig) * uHigh1Theta  , ...
                                cos(phig) * uHigh0Theta ) ;
                Integrand = 1./ sum(InnerMax,2).^2 ;
                Integral = Integrand' * dphig ;

                PiMatHigh(z0,z1) = Integral * uHigh0Theta(z0) * uHigh1Theta(z1) ;
                
            end
            
        end
        
        % Med types
        
        R = ( uMed1Theta(z1) * uMed0Theta(z0) ) / ( uMed0Theta(z1) * uMed1Theta(z0) ) ;
        
        if z0 == z1
            
            PiMatMed(z0,z1) =  uMed0Theta(z0) * uMed1Theta(z0) / sum( max( uMed0Theta(z0) * uMed1Theta , uMed1Theta(z0) * uMed0Theta ) ) ;
            
        elseif z0 ~= z1
            
            if isnan(R) == 0 && R > 1

                                
                phiA = atan( uMed0Theta(z1) / uMed1Theta(z1) ) ;
                phiB = atan( uMed0Theta(z0) / uMed1Theta(z0) ) ;
                
                ls = ( logspace(0,1,NI)' - 1 ) / 9 ;
                
                phi1 = phiA + ( ( phiB + phiA ) / 2 - phiA ) ...
                     * ls ;
                 
                phi2 = ( phiB + phiA ) / 2 + ( phiB - ( phiB + phiA ) / 2 ) ...
                     * ( 1-fliplr(ls')' ) ;
                
                phig = [ phi1 ; phi2 ] ;
                
                dphig = [ phig(2:end) - phig(1:(end-1)) ; phig(end) - phig(end-1) ] ;

                InnerMax = max( sin(phig) * uMed1Theta  , ...
                                cos(phig) * uMed0Theta ) ;
                Integrand = 1./ sum(InnerMax,2).^2 ;
                Integral = Integrand' * dphig ;

                PiMatMed(z0,z1) = Integral * uMed0Theta(z0) * uMed1Theta(z1) ;
                
            end
            
        end
          
     end
end

% compute location changes
PiMatLow(isnan(PiMatLow)) = 0 ;
sum(PiMatLow(:))   

PiMatHigh(isnan(PiMatHigh)) = 0 ;
sum(PiMatHigh(:))  ;
PiMatMed(isnan(PiMatMed)) = 0 ;

dz = ones(Nz,1) * z - z' * ones(1,Nz) ;

dzLowCond  = sum( PiMatLow  .* dz , 2 ) ./ sum( PiMatLow  , 2 ) ;
dzLowCond(isnan(dzLowCond)) = 0 ;
dzHighCond = sum( PiMatHigh .* dz , 2 ) ./ sum( PiMatHigh , 2 ) ;
dzHighCond(isnan(dzHighCond)) = 0 ;
dzMedCond = sum( PiMatMed .* dz , 2 ) ./ sum( PiMatMed , 2 ) ;
dzMedCond(isnan(dzMedCond)) = 0 ;

InitLocLow  = sum( PiMatLow  , 2 ) ;
InitLocHigh = sum( PiMatHigh , 2 ) ;
InitLocMed = sum( PiMatMed , 2 ) ;

% unconditional DiD
dzLow  = sum( PiMatLow(:)  .* dz(:) ) / sum( PiMatLow(:) ) ;
dzHigh = sum( PiMatHigh(:) .* dz(:) ) / sum( PiMatHigh(:) ) ;
dzMed = sum( PiMatMed(:) .* dz(:) ) / sum( PiMatMed(:) ) ;
ddzMean = dzLow - dzHigh ;
ddzMeanMed = dzMed - dzHigh ;
                 
shocks = 'permanent' ;
run FiguresAmenities.m




