


clear all; close all;
dbstop if error

% This code solves the individual problem using endogenous gridpoints
% We impose an exogenous house rent function
% We solve it for one particular skill

tic;
start = toc;


%% GLOBAL PARAMETERS

global gamma Na Nz Ny y R beta amin zmin zmax Ty TyTranspose a da z dz tol
global aStackFL yStackFL aStack yStack zStack smoothP
global eta q0 S Ns ds sDensity zDensity date

date = '' ;


%% ECONOMIC PARAMETERS

gamma   = 5 ;                         % Risk-aversion

smin    = 0.8 ;                       % Lowest skill of individual
smax    = 1.2 ;                       % Highest skill of individual

Ny      = 2 ;                         % Number of income states
y       = [0.05 ; 0.2] ;              % Income states

beta    = 0.95 ;                      % Discount factor
R       = 1.04 ;                      % Interest rate
    
xi      = 0.0;                        % Tightness of constraint relative to natural BC
abar    = 0 ;                         % Borrowing constraint
amin    = abar;                       % Constraint is lower bound on asset grid

amax    = 0.8 ;                       % Highest asset level. If too low EGM does not converge

Ty      = zeros(Ny,Ny);               % Income transition matrix
Ty(2,1) = 0.1;
Ty(1,2) = 0.6; 
Ty(1,1) = 1 - Ty(1,2);                
Ty(2,2) = 1 - Ty(2,1); 

TyTranspose = Ty';

zmax    = 1;                          % Best city
zmin    = 0.3;                        % Worst city

q0      = 0.01;                       % House rent shifter
eta     = 0.5 ;                       % House rent elasticity to population


%% NUMBER OF GRIDPOINTS AND NUMERICAL SETTINGS

% Number of points for EGM
Na = 120;
Nz = 80;
Ns = 80;

% Smoothing parameters
smoothFit = 0.0 ;
smoothNew = 0.0 ;
smoothP   = 0 ; % For EGM updating

% Convergence tolerance criterion
tol = 1e-12;

% Gridpoints for assets
a = linspace(amin,amax,Na)';
da = [a(2) - a(1) ; a(2:end) - a(1:end-1)];

% Gridpoints for city income
z = linspace(zmin,zmax,Nz)';
dz = [z(2) - z(1) ; z(2:end) - z(1:end-1)];

% Gridpoints for skill
S = linspace(smin,smax,Ns)';
if Ns > 1
    ds = [S(2) - S(1) ; S(2:end) - S(1:end-1)];
else 
    ds = 1 ; 
end

% Skill density
sDensity = 1/Ns*ones(Ns,1)./ds;
sMass = 1/Ns*ones(Ns,1);

% Housing density
zDensity = 1/Nz*ones(Nz,1)./dz;
zMass = 1/Nz*ones(Nz,1);

% Stacked grids: a is the innermost dimension, then z, then y
aStack = repmat(a,Nz*Ny,1);
zStack = repmat(kron(z,ones(Na,1)),Ny,1);
yStack = kron(y,ones(Na*Nz,1));

aStackFL = repmat(a,Ny,1);
yStackFL = kron(y,ones(Na,1));


%% INITIAL GUESS FOR HOUSE RENTS

% Construct approximation to long-run MRS in low income state
mrs = ( Ty(1,:)*( ((R-1)*amin + y).^(-gamma))  )/( ((R-1)*amin + y(1)).^(-gamma));
dqLR = beta*S(1)*mrs;

% Parameters for slope
inter = 0.98 ;
slope = 1.5  ;
conv  = 1.5  ;

% Slope and level of housing prices
Dq = inter*dqLR + slope*(S(1)/R - inter/slope*dqLR)*( ( z-zmin )/(zmax-zmin ) ).^conv;
q = cumsum(Dq.*dz);

% Define stacked grids
qStack = repmat(kron(q,ones(Na,1)),Ny,1);
DqStack = repmat(kron(Dq,ones(Na,1)),Ny,1);

% Stacked grids
qStack = repmat(kron(q,ones(Na,1)),Ny,1);
DqStack = repmat(kron(Dq,ones(Na,1)),Ny,1);

% Compute optimal unconstrained location for lowest skill
s = S(1);
[ zU , qU , iZUBot , iZUTop , fracZU ] = ZU( s , q , Dq ) ;
zFL = zU;
qFL = qU;


%% INITIAL GUESS: ENDOGENOUS GRIDPOINTS WITHOUT LOCATION CHOICE

% Start with lowest skill
s = S(1);

% Initial guess for consumption
c0 = max( R*aStackFL + yStackFL + s*zFL - qFL , 0.01 * yStackFL ) ;
c0Mat = reshape(c0,Na,Ny);
cPFL = c0Mat ;
   
% EGM loop
[ cPFL , aPFL ] = PoliciesFL( s , cPFL , zFL , qFL) ; 


%% LOOP OVER SKILL TYPES

% Initialize empty arrays to save policy and distribution
cMat = zeros(Na,Nz,Ny,Ns);
aMat = zeros(Na,Nz,Ny,Ns);
zMat = zeros(Na,Nz,Ny,Ns);
qMat = zeros(Na,Nz,Ny,Ns);
Trans = cell(Ns,1);

fMat = zeros(Na,Nz,Ny,Ns);

for is=1:Ns

    %%% New skill
    s = S(is);
    disp(['is = ' num2str(is,2) ])

    if is==1
        % Use no location asset policy for first s
        cP = reshape(permute(repmat(cPFL,1,1,Nz),[1 3 2]),Na*Nz,Ny) ;
    elseif is==2
        % Use as an initial guess the policy function of the last skill type
        cP = reshape(cP,Na*Nz,Ny) ;
    else
        % Linear extrapolation of two previous terms
        cP = reshape( cMat(:,:,:,is-1) + ( cMat(:,:,:,is-1)-cMat(:,:,:,is-2) ) / (S(is-1)-S(is-2)) * (S(is)-S(is-1)) , Na*Nz,Ny) ;
    end

    %%% EGM loop
    [ cP , aP ,zP , qP ] = Policies( s , q , qStack , Dq , DqStack , cP ) ;

    % Save policies
    cMat(:,:,:,is) = cP ; 
    aMat(:,:,:,is) = aP ;
    zMat(:,:,:,is) = zP ;
    qMat(:,:,:,is) = qP ;
    
    % Distribution
    [ fMass , fDensity , trans ] = Distribution( S(is) , q , Dq , aMat(:,:,:,is) , zMat(:,:,:,is) );

    % Adjust by mass of skill type and save distribution
    fMat(:,:,:,is) = sMass(is)*fMass ;
    
    Trans{is} = trans ;

end


%% REFERENCE s FOR GRAPHS

isRef = max(1,floor(0.5 * Ns)) ;
sRef  = S(isRef) ;


%% FIXED LOCATION POLICIES

% Policies and distribution

% Compute optimal unconstrained location and house prices there
[ zFL , qFL , iZFLBot , iZFLTop , fracZFL ] = ZU( sRef , q , Dq ) ;
 
% Initial guess
c0 = max( R*aStackFL + yStackFL + sRef*zFL - qFL , 0.01 * yStackFL ) ;
cPFL = reshape(c0,Na,Ny);

[ cPFL , aPFL ] = PoliciesFL( sRef , cPFL , zFL , qFL) ; 
cPFL0 = cPFL ;
aPFL0 = aPFL ;

zPFL = repmat(zFL,Na,Nz,Ny) ;
aPFL = permute(repmat(aPFL,1,1,Nz),[1 3 2]) ;
cPFL = permute(repmat(cPFL,1,1,Nz),[1 3 2]) ;

[ ~ , ~ , TransFL ] = Distribution( sRef , q , Dq , aPFL , zPFL );


%% CONSTRUCT FIGURE 3: TIME SERIES

% Get average annual income to normalize assets
assets = repmat(a,1,Nz,Ny,Ns) ;
locations = permute(repmat(z,1,Na,Ny,Ns),[2 1 3 4]) ;
incomes   = permute(repmat(y,1,Na,Nz,Ns),[2 3 1 4]) ;
skills    = permute(repmat(S,1,Na,Nz,Ny),[2 3 4 1]) ;
AverageIncome = sum( fMat(:) .* ( incomes(:) + skills(:) .* locations(:) ) ) ;
AverageWealth = sum( fMat(:) .* assets(:) );

% Wealth quintiles
fAssets = zeros(Na,1);
for ia=1:Na
    fAssets(ia) = sum(sum(sum(squeeze(fMat(ia,:,:,:)))));
end
FAssets = cumsum(fAssets) ;

% Periods for time series
T1LA = 10 ;
T2LA = 5 ;
TLA = T1LA + T2LA ;
timeLA = (1:(1+T1LA+T2LA)) ;

% Starting point
a0 = 0.2 ;

% Compute time series
[ AtLA,ZtLA,SZtLA,YtLA,ItLA,CtLA,qtLA ] = ...
    TimeSeries(Trans,TransFL,cMat,cPFL,q,Dq,T1LA,T2LA,a0,isRef) ;

% Plot figure 3
TimeSeriesPlot(sRef,AtLA(:,1),AtLA(:,2),...
               CtLA(:,1),CtLA(:,2),...
               ZtLA(:,1),YtLA(:,1),...
               qtLA(:,1),qFL,zFL,T1LA,TLA,timeLA,AverageIncome) ;


%% CONSTRUCT FIGURES 4 AND 12: DiD

% Construct quintiles of asssets
iaQ1 = min(find(FAssets >= 0.2 )) ;
iaQ5 = min(find(FAssets > 0.8 )) ;
Q1 = (1:iaQ1)' ;
Q5 = (iaQ5:Na)' ;
Q = [ Q1 ; Q5 ] ;

% Get unconstrained location of reference skill type sRef
[ zU , qU , iZUBot , iZUTop , fracZU ] = ZU( sRef , q , Dq ) ;
IZ = iZUBot ;

% Compute time series
T1 = 3 ;
T2 = 1 ;
time = (1:(1+T1+T2))-1 ;
Tmax = 1+ T1 + T2 ;     
[ At,Atp1,Ztp1,SZt,It,...
  cAt,cAtp1,cZtp1,cSZt,cIt,...
  pAt,pAtp1,pZtp1,pSZt,pIt,...
  Yt ]= ...
         TimeSeriesDiD(fMat,zMat,aMat,Trans,T1,T2,Q1,Q5,IZ,isRef) ; 
     
% Plot figure 4
DiDPlot( Tmax,1,time,It,Atp1,Ztp1,'') ;

% Plot figure 12
DiDLongPlot(Tmax,time,It,Atp1,At,Ztp1,...
                     cIt,cAtp1,cAt,cZtp1,...
                     pIt,pAtp1,pAt,pZtp1) ;

                 
%% CONSTRUCT FIGURE 11: WELFARE

isWelfare = isRef ;
sWelfare = S(isWelfare) ;
[ zU , qU , izUBot , izUTop , fraczU ] = ZU( sWelfare , q , Dq ) ;

% Policy functions at reference skill
cP = cMat(:,:,:,isRef) ;
aP = aMat(:,:,:,isRef) ;
zP = zMat(:,:,:,isRef) ;

% Difference between policy function with and without location asset
Diff = cP - reshape( cPFL , Na , Nz , Ny ) ;

fMassZU = (1-fraczU)*squeeze(fMat(:,izUBot,:,isWelfare)) + fraczU*squeeze(fMat(:,izUTop,:,isWelfare));

% Recover value functions from policy functions
[ cFL,c,czU,VFL,V,VzU ] = ValueFunctions(aPFL0,cPFL0,aP,cP,zP,sWelfare,q,Dq) ; 

% Plot figure 11
WelfarePlots( cP,cPFL0,cFL,czU,0.5,fraczU,izUBot,izUTop,fMass,fMassZU) ;


