%%% AUTOMATED ESTIMATION OF INFLATION MODELS %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Estimate Univariate and Multivariate Core Trend models on the most recent
% vintage of PCE data. Summaries and a report are produced.
%
% Version: 2022 Mar 03 - Matlab R2020a
% MODIFIED BY MW Watson , 12/16/2024
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Clear memory
clear
close all
rng(2022);

% Determine tasks
estimation = true;
date_used  = '202310';

% Set directories
rng(2022)
addpath('functions');
data_path = [pwd filesep 'data' filesep];
res_path  = [pwd filesep 'results' filesep];
fig_path  = [pwd filesep 'figures' filesep];
tab_path  = [pwd filesep 'tables' filesep];
repo_path = [pwd filesep 'reports' filesep];


%% DATA

% Extract data
pce_m59           = load([data_path 'pce_m59_' date_used '.mat']);
date_str          = pce_m59.date_str;
dates             = pce_m59.dates;
sample            = (dates >= datetime(1960, 1, 1)) & (dates <= datetime(1985, 1, 1));
dates             = dates(sample);
labels_short      = pce_m59.labels_short;
infla_agg         = pce_m59.infla_agg(sample, :);
infla_agg_xfe     = pce_m59.infla_agg_xfe(sample, :);
infla_disagg      = pce_m59.infla_disagg(sample, :);
infla_12m_agg     = pce_m59.infla_12m_agg(sample, :);
infla_12m_agg_xfe = pce_m59.infla_12m_agg_xfe(sample, :);
infla_12m_disagg  = pce_m59.infla_12m_disagg(sample, :);
share             = pce_m59.share(sample, :);
share_xfe         = pce_m59.share_xfe(sample, :);
is_xfe            = (mean(share_xfe, 1) > 0.001);
fprintf('- PCE data for %s/%s\n\n', date_str(5:6), date_str(1:4))


% Set dimensions and tail probabilities for interval estimates
T      = length(dates);
n      = size(infla_disagg, 2);
signif = 1/6;

n_h = 12;

% Set up priors and other parameters that are held fixed across the dates
% Set number of MA lags, time-aggregated sectors and dependent sectors
n_lags    = repmat(3, [n, 1]);
is_timeag = false(n, 1); %is_timeag(9) = true;
i_depend  = zeros(n, 1);
% Set estimation settings
settings               = struct();
settings.show_progress = false;
settings.n_draw        = 10;
settings.n_burn        = 10;
settings.n_thin        = 2;
settings.n_lags    = n_lags;
settings.is_timeag = is_timeag;
settings.i_depend  = i_depend;
settings.n_h       = n_h;
% Priors 
% Set theta/lambda/gamma/ps priors
nper          = 12;
ps_mean       = 1-1/(4*nper);
ps_prior_obs  = 10*nper;
prior         = struct();
prior.prec_MA = 0.1;
prior.nu_lam  = 12;
prior.s2_lam  = 0.25^2/60/nper;
prior.nu_gam  = 60;
prior.s2_gam  = 1/60/nper;
prior.a_ps    = ps_mean*ps_prior_obs;
prior.b_ps    = (1-ps_mean)*ps_prior_obs;
prior.nu_gam  = 60;

output_MCT = estimate_MCT_watson(infla_disagg, prior, settings);
save('output_MCT.mat', 'output_MCT');
    