module ziggurat
	implicit none
	
	  real ( kind = 4 ) fn(128)
	  integer ( kind = 4 ) kn(128)
	  real ( kind = 4 ) wn(128)

	  integer ( kind = 4 ) seed
!$omp threadprivate(seed)	  
	  
	  private
	  public rnun, rnnoa, rnchi2, ran_gamma
	  public init_ziggurat
	  
	interface rnnoa
        module procedure rnnoa_v, rnnoa_m
	end interface
	contains
	
	subroutine init_ziggurat(seed0)
		integer	:: seed0,i
		call r4_nor_setup
		!$omp parallel do	
		do i=1,100
			seed=seed0*i
		enddo
		seed=seed0
	end subroutine
	
	subroutine rnun(v)
		real	:: v(:)
		integer	:: i
		do i=1,size(v)
			v(i)=r4_uni(seed)
			if(v(i)==1) v(i)=1.0-epsilon(1.0)
		enddo
	end subroutine
	
	subroutine rnchi2(df,v)
		real	:: v(:),df
		integer	:: i
		do i=1,size(v)
			v(i)=2*ran_gamma(0.5*df)
		enddo
	end subroutine

	subroutine rnnoa_v(v)
		real	:: v(:)
		integer	:: i
		do i=1,size(v)
			v(i)=r4_nor(seed)
		enddo
	end subroutine

	subroutine rnnoa_m(m)
		real	:: m(:,:)
		integer	:: i
		do i=1,size(m,2)
			call rnnoa_v(m(:,i))
		enddo
	end subroutine
	
	function r4_uni(jsr)
	  integer ( kind = 4 ) jsr
	  integer ( kind = 4 ) jsr_input
	  real ( kind = 4 ) r4_uni

	  jsr_input = jsr

	  jsr = ieor ( jsr, ishft ( jsr,   13 ) )
	  jsr = ieor ( jsr, ishft ( jsr, - 17 ) )
	  jsr = ieor ( jsr, ishft ( jsr,    5 ) )

	  r4_uni = 0.5E+00 + real ( jsr_input + jsr, kind = 4 ) &
		/ real ( 65536, kind = 4 ) / real ( 65536, kind = 4 )

	  return
	end
	
	function shr3(jsr) result(val)
	  integer ( kind = 4 ) jsr
	  integer ( kind = 4 ) jsr_input
	  integer ( kind = 4 ) val

	  jsr_input = jsr

	  jsr = ieor ( jsr, ishft ( jsr,   13 ) )
	  jsr = ieor ( jsr, ishft ( jsr, - 17 ) )
	  jsr = ieor ( jsr, ishft ( jsr,    5 ) )

	  val = jsr_input + jsr
	end
	
	function r4_nor ( jsr) result(value)

	!*****************************************************************************80
	!
	!! R4_NOR returns a normally distributed single precision real value.
	!
	!  Discussion:
	!
	!    The value returned is generated from a distribution with mean 0 and
	!    variance 1.
	!
	!    The underlying algorithm is the ziggurat method.
	!
	!    Before the first call to this function, the user must call R4_NOR_SETUP
	!    to determine the values of KN, FN and WN.
	!
	!    Note that JSR and KN should actually be of unsigned integer type, but
	!    this data type is not supported by FORTRAN90.
	!
	!  Licensing:
	!
	!    This code is distributed under the GNU LGPL license.
	!
	!  Modified:
	!
	!    04 May 2008
	!
	!  Author:
	!
	!    Original C version by George Marsaglia, Wai Wan Tsang.
	!    FORTRAN90 version by John Burkardt.
	!
	!  Reference:
	!
	!    George Marsaglia, Wai Wan Tsang,
	!    The Ziggurat Method for Generating Random Variables,
	!    Journal of Statistical Software,
	!    Volume 5, Number 8, October 2000, seven pages.
	!
	!  Parameters:
	!
	!    Input/output, integer ( kind = 4 ) JSR, the seed.
	!
	!    Input, integer ( kind = 4 ) KN(128), data computed by R4_NOR_SETUP.
	!
	!    Input, real ( kind = 4 ) FN(128), WN(128), data computed by R4_NOR_SETUP.
	!
	!    Output, real ( kind = 4 ) R4_NOR, a normally distributed random value.
	!
	  implicit none

	  integer ( kind = 4 ) hz
	  integer ( kind = 4 ) iz
	  integer ( kind = 4 ) jsr
	  real ( kind = 4 ), parameter :: r = 3.442620E+00
	  real ( kind = 4 ) value
	  real ( kind = 4 ) x
	  real ( kind = 4 ) y

	  hz = shr3(jsr)
	  iz = iand ( hz, 127 )

	  if ( abs ( hz ) < kn(iz+1) ) then

		value = real ( hz, kind = 4 ) * wn(iz+1)

	  else

		do

		  if ( iz == 0 ) then

			do
			  x = - 0.2904764E+00 * log ( r4_uni ( jsr ) )
			  y = - log ( r4_uni ( jsr ) )
			  if ( x * x <= y + y ) then
				exit
			  end if
			end do

			if ( hz <= 0 ) then
			  value = - r - x
			else
			  value = + r + x
			end if

			exit

		  end if

		  x = real ( hz, kind = 4 ) * wn(iz+1)

		  if ( fn(iz+1) + r4_uni ( jsr ) * ( fn(iz) - fn(iz+1) ) &
			 < exp ( - 0.5E+00 * x * x ) ) then
			value = x
			exit
		  end if

		  hz = shr3 ( jsr )
		  iz = iand ( hz, 127 )

		  if ( abs ( hz ) < kn(iz+1) ) then
			value = real ( hz, kind = 4 ) * wn(iz+1)
			exit
		  end if

		end do

	  end if
	end

	subroutine r4_nor_setup ()

	!*****************************************************************************80
	!
	!! R4_NOR_SETUP sets data needed by R4_NOR.
	!
	!  Discussion:
	!
	!    Note that KN should actually be of unsigned integer type, but
	!    this data type is not supported by FORTRAN90.
	!
	!  Licensing:
	!
	!    This code is distributed under the GNU LGPL license.
	!
	!  Modified:
	!
	!    04 May 2008
	!
	!  Author:
	!
	!    Original C version by George Marsaglia, Wai Wan Tsang.
	!    FORTRAN90 version by John Burkardt.
	!
	!  Reference:
	!
	!    George Marsaglia, Wai Wan Tsang,
	!    The Ziggurat Method for Generating Random Variables,
	!    Journal of Statistical Software,
	!    Volume 5, Number 8, October 2000, seven pages.
	!
	!  Parameters:
	!
	!    Output, integer ( kind = 4 ) KN(128), data needed by R4_NOR.
	!
	!    Output, real ( kind = 4 ) FN(128), WN(128), data needed by R4_NOR.
	!
	  implicit none

	  real ( kind = 8 ) dn
	  integer ( kind = 4 ) i
	  real ( kind = 8 ), parameter :: m1 = 2147483648.0D+00
	  real ( kind = 8 ) q
	  real ( kind = 8 ) tn
	  real ( kind = 8 ), parameter :: vn = 9.91256303526217D-03

	  dn = 3.442619855899D+00
	  tn = 3.442619855899D+00

	  q = vn / exp ( - 0.5D+00 * dn * dn )

	  kn(1) = int ( ( dn / q ) * m1 )
	  kn(2) = 0

	  wn(1) = real ( q / m1, kind = 4 )
	  wn(128) = real ( dn / m1, kind = 4 )

	  fn(1) = 1.0E+00
	  fn(128) = real ( exp ( - 0.5D+00 * dn * dn ), kind = 4 )

	  do i = 127, 2, -1
		dn = sqrt ( - 2.0D+00 * log ( vn / dn + exp ( - 0.5D+00 * dn * dn ) ) )
		kn(i+1) = int ( ( dn / tn ) * m1 )
		tn = dn
		fn(i) = real ( exp ( - 0.5D+00 * dn * dn ), kind = 4 )
		wn(i) = real ( dn / m1, kind = 4 )
	  end do

	  return
	end

	function ran_gamma(s) result(fn_val)
		! uses the algorithm in
		! marsaglia, g. and tsang, w.w. (2000) `a simple method for generating
		! gamma variables', trans. om math. software (toms).

		! generates a random gamma deviate for shape parameter s >= 1.
		real, intent(in)    :: s
		real                :: fn_val

		! local variables
		real		:: c, d
		real        :: u, v, x

		if (s < 1.0) then
		   write(*, *) 'shape parameter must be >= 1'
		   stop
		end if

		d = s - 1./3.
		c = 1.0/sqrt(9.0*d)

		! start of main loop
		do

		! generate v = (1+cx)^3 where x is random normal; repeat if v <= 0.

		  do
			x = r4_nor(seed)
			v = (1.0 + c*x)**3
			if (v > 0.0) exit
		  end do

		! generate uniform variable u

		  u=r4_uni(seed)
		  if (u < 1.0 - 0.0331*x**4) then
			fn_val = d*v
			exit
		  else if (log(u) < 0.5*x**2 + d*(1.0 - v + log(v))) then
			fn_val = d*v
			exit
		  end if
		end do
	end function ran_gamma
	
end module