% Construct GVAR forecasts for State Employment
clear all;
small = 1.0e-8;
big = 1.0e+8;
this_date = datestr(now,'yyyymmdd');

% -- File Directories  
datadir='../Data/'; 
outdir = 'out/';
figdir = 'fig/';
matdir = 'mat/';

% Add paths for Matlab Utility Functions
addpath('../Matlab_Utility_Programs/');

% Parameters for the exercise
n_arlags = 12;  % Number of AR lags
n_agglags = 6;  % Number of lags for the aggregate 
i_constant = 1; % Include a constant term
first_fcst = [1999 12]; % First forecast period
n_h = 12; % Number of forecast periods

% Read Employment Data
SData = readtable([datadir 'StateEmployment.xlsx']);
% Get the state names
states = SData.Properties.VariableNames(3:end);
% Get the data
nper = 12; 
[dnobs,calvec,calds] = calendar_make([SData.Year(1) SData.Month(1)],[SData.Year(end) SData.Month(end)],nper);
% Number of states
nstates = length(states);
dnobs = length(calvec);

% Matrices for saving the results
rslt_fcsts = NaN(n_h,nstates,dnobs);      % Forecasted values
rslt_actual = NaN(n_h,nstates,dnobs);     % Actual values
sd_fcsts = NaN(n_h,nstates,dnobs);        % SD of forecast
sd_fcsts_accum = NaN(n_h,nstates,dnobs);  % STD of forecast accumulated values

% State Employment Data
lev_data = SData{:,3:end};
lfd_data = dif(log(lev_data),1);
lfd_agg = mean(lfd_data,2);

% Find first forecast period .. find first_fcst in calds
t_first = find(calds(:,1) == first_fcst(1) & calds(:,2) == first_fcst(2));

for t = t_first:dnobs
    % Matrices for holding estimates
    phi_ar_mat = zeros(n_arlags,nstates);
    const_mat = zeros(nstates,1);
    if n_agglags > 0
        phi_agg_mat = zeros(n_agglags,nstates);
    else 
        phi_agg_mat = zeros(1,nstates);
    end
    cov_res_mat = zeros(nstates,nstates);
    res_mat = NaN(t,nstates);
    % Estimate the GVAR equation for each state
    for i_state = 1:nstates
        tmp_ar_data = lfd_data(1:t,i_state);
        tmp_agg_data = lfd_agg(1:t);
        [phi_ar,phi_agg,constant,res] = gvar_c_residual(tmp_ar_data,tmp_agg_data,n_arlags,n_agglags,i_constant);
        phi_ar_mat(:,i_state) = phi_ar;
        phi_agg_mat(:,i_state) = phi_agg;
        const_mat(i_state) = constant;
        res_mat(:,i_state) = res;
    end
    % Construct the covariance matrix of the residuals
    e_mat = packr(res_mat);
    ndf = size(e_mat,1)-n_arlags -n_agglags-i_constant; % Degrees of freedom
    cov_res = (e_mat'*e_mat)/ndf;
    % Construct GVAR State Space Model
    [s,phi,theta] = gvar_withconstant_to_ss(phi_ar_mat,phi_agg_mat,const_mat,cov_res);
    n_state = size(phi,1);
    % Forecast -- levels
    % State Vector
    x = zeros(n_state,1);
    x(end) = 1;  % Constant term
    for i_lag = 1:n_arlags
        x(nstates*(i_lag-1)+1:nstates*i_lag) = lfd_data(t+1-i_lag,:)';
    end
    if n_agglags > 0
        for i_lag = 1:n_agglags
            x(nstates*n_arlags+i_lag) = lfd_agg(t+1-i_lag);
        end
    else
        x(nstates*n_arlags+1) = 0;  
    end
    for i_h = 1:n_h
            x = phi*x;
            rslt_fcsts(i_h,:,t) = s*x;
            if (t+i_h <= dnobs)
                rslt_actual(i_h,:,t) = lfd_data(t+i_h,:);
            end
    end
    
    % Comput Covariance Matrix of forecast errors
    P_h = zeros(n_state,n_state);
    Sigma_Fcst_Errors = SS_Model_Fcst_Error_Covmatrix(s,phi,theta,n_h,P_h);
    % Isolate the forecast errors for each entity
    for i_state = 1:nstates
        B = zeros(n_h,n_h*nstates);
        for i_h = 1:n_h
            B(i_h,(i_h-1)*nstates+i_state) = 1;
        end
        Sigma_Fcst_Errors_i = B*Sigma_Fcst_Errors*B';
        sd_fcsts(:,i_state,t) = sqrt(diag(Sigma_Fcst_Errors_i));
        % Accumulated values
        A = tril(ones(n_h,n_h));
        sd_fcsts_accum(:,i_state,t) = sqrt(diag(A*Sigma_Fcst_Errors_i*A'));
    end
end

% Save the results
% Variables to save
var_save = {'rslt_fcsts','rslt_actual','sd_fcsts','sd_fcsts_accum','calvec','calds','dnobs','states'};
save([matdir 'gvar_fcst_state_emp.mat'],var_save{:});
