function [s,phi,theta] = gvar_withconstant_to_ss(rho_ar_mat,rho_agg_mat,constant,cov_mat)
    % Convert to GVAR model to a SS model of the form
    % h(t) = phi*h(t-1) + theta*u(t)
    % y(t) = s*h(t)
    % where h(t) is the state variable, u(t) is the innovation with unit variance, and y(t) is the observation
    % rho_ar_mat are the AR coefficients
    % rho_agg_mat are the coefficients on the aggregates
    % constant is the constant term -- added as the last element of the state 
    % cov_mat is the covariance matrix of the residuals from the GVAR 
    % Let x(t) denote the vector of entities in the gvar
    % Let z(t) = mean(x(t)) denote the aggregate
    % The GVAR is
    % x(t) = rho_ar1*x(t-1) + ... + rho_arn*x(t-n_ar) + rho_agg1*z(t-1) + ... + rho_aggn*z(t-n_agg) + constant + e(t)
    % where e(t) ~ N(0,cov_mat)
    % The state vector is
    % h(t) = [x(t); x(t-1); ...; x(t-n_ar+1); z(t); z(t-1); ... z(t-n_agg+1); constant]
    % Set up the SS matrices
    n_ent = size(rho_ar_mat,2); % Number of entities
    n_ar_lag = size(rho_ar_mat,1); % Number of AR lags
    n_ag_lag = size(rho_agg_mat,1); % Number of aggregate
    n_states = n_ent*n_ar_lag + n_ag_lag + 1; % Number of states
    s = zeros(n_ent,n_states);
    phi = zeros(n_states,n_states);
    theta = zeros(n_states,n_ent);
    % Observation matrix
    s(1:n_ent,1:n_ent) = eye(n_ent);
    theta(1:n_ent,1:n_ent) = chol(cov_mat,'lower');
    % Phi Matrix
    % lags of x
    for i_lag = 2:n_ar_lag
        phi(n_ent*(i_lag-1)+1:n_ent*(i_lag),n_ent*(i_lag-2)+1:n_ent*(i_lag-1)) = eye(n_ent);
    end
    % first lag of z
    % lags of z
    for i_lag = 2:n_ag_lag
        phi(n_ent*n_ar_lag+i_lag,n_ent*n_ar_lag+i_lag-1) = 1;
    end
    % constant term
    phi(end,end) = 1;
    for i_lag = 1:n_ar_lag
        tmp = diag(rho_ar_mat(i_lag,:));
        phi(1:n_ent,(i_lag-1)*n_ent+1:i_lag*n_ent) = tmp;
    end
    for i_lag = 1:n_ag_lag
        phi(1:n_ent,n_ent*n_ar_lag+i_lag) = rho_agg_mat(i_lag,:)'; ;
    end
    phi(1:n_ent,end) = constant;

    % Adjust so that z(t) is the mean of x(t)
    A = eye(n_states);
    A(n_ent*n_ar_lag+1,1:n_ent) = -1/n_ent;

    Ai = inv(A);
    phi = Ai*phi;
    theta = Ai*theta;
end
    
