rm(list = ls())
gc()

# Set your directory to the folder "../ISVM_Code"
setwd("../ISVM_Code")

# Load required packages
require(rmatio)
require(ggplot2)
require(gridExtra)
require(np)

# Source functions
source("Nonparametric_ISVM/data_pre.R")
source("Nonparametric_ISVM/SVestimate.R")

############################
#### Part I: Estimation ####
############################

#### This is one-trial estimation ####

data = data_pre("20070101", "20171231") # Data preprocessing
n = data$n
beta01 = data$beta01; beta02 = data$beta02; beta10 = data$beta10
v = data$v; x_all = data$x_all; t_all = data$t_all; k_all = data$k_all
r = data$r; d = data$d; numofIV = data$numofIV

# Combine the proprocessed data
dat = matrix(c(v, beta01, beta02, beta10, r, d), nrow = n)

# Nonparametric estimation
b = SVestimate(dat) # This plot corresponds to Figure 1 of the code guide file without the red dashed curves for bootstrap CI

# Save the estimation results
v0 = b[1 : n]
sigma10 = b[(n + 1) : (2*n)]
Sigma10 = b[(2*n + 1) : (3*n)]
sigma220 = b[(3*n + 1) : (4*n)]
Sigma220 = b[(4*n + 1) : (5*n)]
Sigma20 = b[(5*n + 1) : (6*n)]
mu0 = b[(6*n + 1) : (7*n)]
Mu0 = b[(7*n + 1) : (8*n)]
Rho0 = b[(8*n + 1) : (9*n)]
dsigma10 = b[(9*n + 1) : (10*n)]

############################
#### Part II: Bootstrap #### 
############################

#### Construct the confidence intervals by bootstrap ####
#### To save time for bootstrap computing, one can comment the plotting commands in lines 66-71 of the function "SVestimate.R" ####

# Initialization
numoftrial = 500 # Number of bootstrap trials for calculating standard errors
vboot = NULL; rboot = NULL; dboot = NULL;
beta01boot = NULL; beta02boot = NULL; beta10boot = NULL;
v_all = NULL; sigma1_all = NULL; sigma2_all = NULL; sigma22_all = NULL; mu_all = NULL; rho_all = NULL

# Bootstrap
for (boottrial in 1 : numoftrial){
  # In each bootstrap trial, we resample IV on each day
  for (i in 1 : n){
    vt = v0[i]
    k_trial0 = k_all[[i]]; t_trial0 = t_all[[i]]; x_trial0 = x_all[[i]] # The original data on the ith day
    
    # Randomly select IV from the original IV surface
    repeat{
      trial_ind = sample(1 : length(x_trial0), numofIV[i], replace = T)
      t_trial = t_trial0[trial_ind]
      k_trial = k_trial0[trial_ind]
      x_trial = x_trial0[trial_ind]
      if (length(unique(t_trial)) >= 4) break
    }
    
    # Construct the regressors
    k2 = k_trial^2
    tk = t_trial*k_trial
    t2 = t_trial^2
    t2k = t_trial^2*k_trial
    
    # Bivariate regression
    obj = lm(x_trial ~ k_trial + k2 + t_trial + tk + t2 + t2k)
    vboot[i] = obj$coefficients[1]
    beta01boot[i] = obj$coefficients[2]
    beta02boot[i] = obj$coefficients[3]
    beta10boot[i] = obj$coefficients[4]
    rboot[i] = r[i]
    dboot[i] = d[i]
  }
  
  ind = which(is.nan(vboot))
  if (length(ind) > 0){
    beta01boot = beta01boot[-ind]
    beta02boot = beta02boot[-ind]
    beta10boot = beta20boot[-ind]
    rboot = rboot[-ind]
    dboot = dboot[-ind]
    vboot = vboot[-ind]
  }
  
  # Nonparametric estimation based on the bootstrap data
  boot = SVestimate(matrix(c(vboot, beta01boot, beta02boot, beta10boot, rboot, dboot), nrow = length(vboot)))
  N = length(vboot)
  v = boot[1 : N]
  sigma1 = boot[(N + 1) : (2*N)]
  Sigma1 = boot[(2*N + 1) : (3*N)]
  sigma22 = boot[(3*N + 1) : (4*N)]
  Sigma22 = boot[(4*N + 1) : (5*N)]
  Sigma2 = boot[(5*N + 1) : (6*N)]
  mu = boot[(6*N + 1) : (7*N)]
  Mu = boot[(7*N + 1) : (8*N)]
  Rho = boot[(8*N + 1) : (9*N)]
  dsigma1 = boot[(9*N + 1) : (10*N)]
  
  # Save the nonparametric estimators
  v_all[boottrial] = list(v); sigma1_all[boottrial] = list(Sigma1); sigma2_all[boottrial] = list(Sigma2)
  sigma22_all[boottrial] = list(Sigma22); mu_all[boottrial] = list(Mu); rho_all[boottrial] = list(Rho)
}

#### Plot the estimation result with CI ####

# x includes 50 points in the range of v0. We use this range for the plot
x = max(min(v0), 0.07) + (0:49)/49*(min(max(v0), 0.65) - max(min(v0), 0.07))

# Initialization
Sigma1_pre = matrix(nrow = length(x), ncol = numoftrial)
Sigma2_pre = matrix(nrow = length(x), ncol = numoftrial)
Sigma22_pre = matrix(nrow = length(x), ncol = numoftrial)
Mu_pre = matrix(nrow = length(x), ncol = numoftrial)
Rho_pre = matrix(nrow = length(x), ncol = numoftrial)

# Use smoothing spline to interpolate the unknown functions at x
for (i in 1 : numoftrial){
  plot_index = (v_all[[i]] <= 0.65 & v_all[[i]] >= 0.07) # For 2007 - 2017
  Sigma1_pre[1 : length(x), i] = predict(smooth.spline(v_all[[i]][plot_index], sigma1_all[[i]][plot_index],
                                                       df = 5, spar = 0.5), x)$y
  Sigma2_pre[1 : length(x), i] = predict(smooth.spline(v_all[[i]][plot_index], sigma2_all[[i]][plot_index],
                                                       df = 5, spar = 0.5), x)$y
  Sigma22_pre[1 : length(x), i] <- predict(smooth.spline(v_all[[i]][plot_index], sigma22_all[[i]][plot_index],
                                                         df = 5, spar = 0.5), x)$y
  Mu_pre[1 : length(x), i] = predict(smooth.spline(v_all[[i]][plot_index], mu_all[[i]][plot_index],
                                                   df = 5, spar = 0.5), x)$y
  Rho_pre[1 : length(x), i] = predict(smooth.spline(v_all[[i]][plot_index], rho_all[[i]][plot_index],
                                                    df = 5, spar = 0.5), x)$y
}

# Calculate the standard deviations of the bootstrap estimators
Sigma1_sd = rep(0,length(x)); Sigma2_sd = rep(0,length(x)); Sigma22_sd = rep(0,length(x))
Mu_sd = rep(0,length(x)); Rho_sd = rep(0,length(x))
for (i in 1 : length(x)){
  Sigma1_sd[i] = sd(Sigma1_pre[i, 1 : numoftrial][which(!is.nan(Sigma1_pre[i, 1 : numoftrial]))])
  Sigma2_sd[i] = sd(Sigma2_pre[i, 1 : numoftrial][which(!is.nan(Sigma2_pre[i, 1 : numoftrial]))])
  Sigma22_sd[i] = sd(Sigma22_pre[i, 1 : numoftrial][which(!is.nan(Sigma22_pre[i, 1 : numoftrial]))])
  Mu_sd[i] = sd(Mu_pre[i, 1 : numoftrial][which(!is.nan(Mu_pre[i, 1 : numoftrial]))])
  Rho_sd[i] = sd(Rho_pre[i, 1 : numoftrial][which(!is.nan(Rho_pre[i, 1 : numoftrial]))])
}

# Calculate the unknown functions at x based on the original (not bootstrap) nonparametric estimators
plot_index = (v0 <= 0.65 & v0 >= 0.07)
Sigma10_pre = predict(smooth.spline(v0[plot_index], Sigma10[plot_index], df = 5, spar = 0.5), x)$y;
Sigma20_pre = predict(smooth.spline(v0[plot_index], Sigma20[plot_index], df = 5, spar = 0.5), x)$y;
Sigma220_pre = predict(smooth.spline(v0[plot_index], Sigma220[plot_index], df = 5, spar = 0.5), x)$y;
Mu0_pre = predict(smooth.spline(v0[plot_index], Mu0[plot_index], df = 5, spar = 0.5), x)$y;
Rho0_pre = predict(smooth.spline(v0[plot_index], Rho0[plot_index], df = 5, spar = 0.5), x)$y;

# Calculate CI (estimator +/- 2*standard error)
Sigma1_u <- Sigma10_pre + 2*Sigma1_sd; Sigma1_d <- Sigma10_pre - 2*Sigma1_sd;
Sigma2_u <- Sigma20_pre + 2*Sigma2_sd; Sigma2_d <- Sigma20_pre - 2*Sigma2_sd;
Sigma22_u <- Sigma220_pre + 2*Sigma22_sd; Sigma22_d <- Sigma220_pre - 2*Sigma22_sd;
Mu_u <- Mu0_pre + 2*Mu_sd; Mu_d <- Mu0_pre - 2*Mu_sd;
Rho_u <- Rho0_pre + 2*Rho_sd; Rho_d <- Rho0_pre - 2*Rho_sd;

# Final plot with CI
p1 = qplot(v0[plot_index], sigma10[plot_index], xlab = "v", ylab = "gamma(v)") + geom_line(aes(x, Sigma10_pre), col = 'red') + geom_line(aes(x, Sigma1_u), col = 'red', linetype = 'dashed') + geom_line(aes(x, Sigma1_d), col = 'red', linetype = 'dashed')
p2 = qplot(v0[plot_index], sigma220[plot_index], xlab = "v", ylab = "eta(v)^2") + geom_line(aes(x, Sigma220_pre), col = 'red') + geom_line(aes(x, Sigma22_u), col = 'red', linetype = 'dashed') + geom_line(aes(x, Sigma22_d), col = 'red',linetype = 'dashed')
p3 = qplot(xlab = "v", ylab = "eta(v)") + geom_line(aes(x, Sigma20_pre), col = "red") + geom_line(aes(x, Sigma2_u), col = 'red', linetype = 'dashed') + geom_line(aes(x, Sigma2_d), col = 'red', linetype = 'dashed')
p4 = qplot(v0[plot_index], mu[plot_index], xlab = "v", ylab = "mu(v)") + geom_line(aes(x, Mu0_pre), col = 'red') + geom_line(aes(x, Mu_u), col = 'red', linetype = 'dashed') + geom_line(aes(x, Mu_d), col = 'red', linetype = 'dashed')
p5 = qplot(xlab = "v", ylab = "rho(v)") + geom_line(aes(x, Rho0_pre), col = "red") + geom_line(aes(x, Rho_u), col = 'red',linetype = 'dashed') + geom_line(aes(x, Rho_d), col = 'red', linetype = 'dashed')
grid.arrange(p1, p2, p3, p4, p5, nrow = 2) # This plot corresponds to Figure 1 of the code guide file with CI