SVestimate = function(dat){
  
  #### Input ####
  # dat: The data of expansion terms (shape characteristics), risk-free interest rates, dividend yields
  #### Ouput ####
  # v: The spot volatilities
  # sigma1: The data of the function gamma (see equation (20))
  # Sigma1: The nonparametric estimator of the function gamma
  # sigma22: The data of the function eta^2 (see euation (25))
  # Sigma22: The nonparametric estimator of the function eta^2
  # Sigma2: The nonparametric estimator of eta
  # mu: The data of the function mu (see equation (26))
  # Mu: The nonparametric estimator of the function mu
  # Rho: The nonparametric estimator of the leverage effect function rho (see equation (35))
  # dsigma1: The local linear estimator of the first order derivative of gamma (see equation (24))
  
  # Extract items from dat
  v = dat[,1];
  beta01 = dat[,2];
  beta02 = dat[,3];
  beta10 = dat[,4];
  r = dat[,5];
  d = dat[,6];
  
  # Calculate the data of gamma
  sigma1 = 2*v*beta01 # Equation (20)
  
  # Set the bandwidth in the nonparametric estimation of gamma
  bw1 = 0.15 # You may adjust the bandwidth for different sample period
  
  # Local linear nonparametric estimation of gamma
  fit1 = npreg(bws = bw1, txdat = v, tydat = sigma1, ckertype = 'epanechnikov', 
               regtype = 'll', bwtype = 'fixed', gradients = TRUE)
  dsigma1 = fit1$grad[,] # Equation (24)
  Sigma1 = fit1$mean # Equation (23)
  
  # Calculate the data of eta^2
  sigma22 = 3*v^3*beta02*2 - v*Sigma1*dsigma1 + 3/2*Sigma1^2 # Equation (25)
  
  # Set the bandwidth in the nonparametric estimation of eta^2
  bw2 = 0.16 # You may adjust the bandwidth for different sample period
  
  # Local linear nonparametric regression of eta^2
  fit2 = npreg(bws = bw2, txdat = v, tydat = sigma22, ckertype = 'epanechnikov',
               regtype = 'll', bwtype = 'fixed', gradients = TRUE)
  Sigma22 = fit2$mean
  Sigma2 = sqrt(Sigma22)
  
  # Calculate the data of mu
  mu = 1/12/v*(24*v*beta10 - 2*Sigma1*(6*(d - r) - 2*v*dsigma1 + 3*v^2) - 3*Sigma1^2 - 2*Sigma22) # Equation (26)
  
  # Set the bandwidth in the nonparametric estimation of mu
  bw3 = 0.15 # You may adjust the bandwidth for different sample period
  
  # Local linear nonparametric estimation of mu
  fit3 = npreg(bws = bw3, txdat = v, tydat = mu, ckertype = 'epanechnikov', 
               regtype = 'll', bwtype = 'fixed')
  Mu = fit3$mean
  
  Rho=Sigma1/sqrt(Sigma1^2+Sigma22)
  
  # Set the range of plots
  plot_index = (v <= 0.65 & v >= 0.07)
  
  # Plot the estimated functions
  p1 = qplot(v[plot_index], sigma1[plot_index], xlab = "v", ylab = "gamma(v)") + geom_line(aes(v[plot_index], Sigma1[plot_index]), col = 'red')
  p2 = qplot(v[plot_index], sigma22[plot_index], xlab = "v", ylab = "eta(v)^2") + geom_line(aes(v[plot_index], Sigma22[plot_index]), col='red')
  p3 = qplot(v[plot_index], Sigma2[plot_index], geom = 'line', xlab = "v", ylab = "eta(v)")
  p4 = qplot(v[plot_index], mu[plot_index], xlab = "v", ylab = "mu(v)") + geom_line(aes(v[plot_index], Mu[plot_index]), col='red')
  p5 = qplot(v[plot_index], Rho[plot_index], geom = 'line', xlab = "v", ylab = "rho(v)")
  grid.arrange(p1, p2, p3, p4, p5, nrow = 2)
  
  # Combine and return the results
  return(c(v, sigma1, Sigma1, sigma22, Sigma22, Sigma2, mu, Mu, Rho, dsigma1))
}