function output = TwoStepGMM(r, d, param_initial, v, beta01, beta10, beta11, beta02, beta20, ...
                             std_01, std_10, std_11, std_02, std_20)
    %%%% This function performs GMM estimation %%%%
    %%%% Input %%%%
    % r: Vector of interest rate
    % d: Vector of dividend
    % param_initial: Vector of initial parameters [kappa, alpha, xi, rho] for optimization
    % v: Vector of spot volatilities
    % beta01: Data beta_01
    % beta10: Data beta_10
    % beta11: Data beta_11
    % beta02: Data beta_02
    % beta20: Data beta_20
    % std_01: Standard derivation of data beta_01
    % std_10: Standard derivation of data beta_10
    % std_11: Standard derivation of data beta_11
    % std_02: Standard derivation of data beta_02
    % std_20: Standard derivation of data beta_20
    %%%% Output %%%%
    % output: The estimators of model parameters [kappa, alpha, xi, rho]
    
    % A function to calculate the sum of Omega_j + Omega_j' for all j >= 1
    function Omegaj = element(mag, j)
       Sw = zeros(size(mag, 1), size(mag, 1));
       for i = (j + 1) : size(mag, 2)
           Sw = Sw + mag(:, i)*mag(:, i - j)'; % See the equation below (C2)
       end
       Sw = Sw/size(mag, 2);
       Omegaj = Sw + Sw';
    end

    % Two-step GMM estimation
    % Step 1: Estimate the weight matrix W_n in (28)
    w1 = eye(5);
    options = optimoptions('fmincon', 'MaxFunctionEvaluations', 50000, 'MaxIterations', 5000, 'display','off');
    objfun = @(params)(GMM_loss_fun(r, d, [params(1) params(2) params(3) params(4)], v, beta01, beta10, beta11, beta02, beta20, w1, ...
                                    std_01, std_10, std_11, std_02, std_20)); % GMM loss function without Feller constraint
    param_first = fmincon(objfun, param_initial, [], [], [], [], [0 0 0 -1], [Inf Inf Inf 1], [], options);
    if (2*param_first(1)*param_first(2) - param_first(3)*param_first(3) < 0) % If the Feller condition is violated, add Feller condition as the constraint
        param_init = [param_initial(1 : 2), param_initial(4)];
        objfun = @(params)(GMM_loss_fun_cons(r, d, [params(1) params(2) params(3)], v, beta01, beta10, beta11, beta02, beta20, w1,...
                                             std_01, std_10, std_11, std_02, std_20)); % GMM loss function with Feller constraint
        param_first = fmincon(objfun, param_init, [], [], [], [], [0 0 -1], [Inf Inf 1], [], options);
        param_first = [param_first(1 : 2), sqrt(2*param_first(1)*param_first(2)), param_first(3)]; % The first estimates of four model parameters
    end

    % Calculate \hat{\Omega}
    mag = moment_fun('over_g', r, d, param_first, v, beta01, beta10, beta11, beta02, beta20, ...
                      std_01, std_10, std_11, std_02, std_20);
    nlag = round(0.75*size(mag, 2)^(1/3) - 1); % Lag \ell in equation (C2)
    % Calculate Omega_0
    S = zeros(size(mag, 1), size(mag, 1));
    for k = 1:size(mag,2)
       S = S + mag(:, k)*mag(:, k)';
    end
    Omega0 = S/size(mag, 2);
    Oemga_hat = Omega0;
    for j = 1 : nlag
       Oemga_hat = Oemga_hat + (nlag + 1 - j)/(nlag + 1)*element(mag, j); % See equation (C2)
    end
    % The optimal weight matrix W_n in (28)
    w2 = inv(Oemga_hat);
    
    % Step 2: Estimate model parameters based on the optimal weight matrix 
    objfun2 = @(params)(GMM_loss_fun(r, d, [params(1) params(2) params(3) params(4)], v, beta01, beta10, beta11, beta02, beta20, w2, ...
                                     std_01, std_10, std_11, std_02, std_20)); % GMM loss function without Feller constraint
    output = fmincon(objfun2, param_first, [], [], [], [], [0 0 0 -1], [Inf Inf Inf 1], [], options);
    if (2*output(1)*output(2) - output(3)*output(3) < 0) % If the Feller condition is violated, add Feller condition as the constraint
        param_cons = [param_first(1 : 2), param_first(4)];
        objfun2 = @(params)(GMM_loss_fun_cons(r, d, [params(1) params(2) params(3)], v, beta01, beta10, beta11, beta02, beta20, w2,...
                                              std_01, std_10, std_11, std_02, std_20)); % GMM loss function with Feller constraint
        output_cons = fmincon(objfun2, param_cons, [], [], [], [], [0 0 -1], [Inf Inf 1], [], options);
        output = [output_cons(1 : 2), sqrt(2*output_cons(1)*output_cons(2)), output_cons(3)]; % The final estimates of four model parameters
    end
end