function output = asy_var(type, r, d, param, v, beta01, beta10, beta11, beta02, beta20, ...
                                std_01, std_10, std_11, std_02, std_20)
    %%%% This function is used for calculating the asymptotic variance matrix V/n in equation (29) %%%%
    %%%% Input %%%%
    % r: Vector of interest rate
    % d: Vector of dividend
    % param: Vector of model parameters [kappa, alpha, xi, rho]    
    % v: Vector of spot volatilities
    % beta01: Data beta_01
    % beta10: Data beta_10
    % beta11: Data beta_11
    % beta02: Data beta_02
    % beta20: Data beta_20
    % std_01: Standard derivation of data beta_01
    % std_10: Standard derivation of data beta_10
    % std_11: Standard derivation of data beta_11
    % std_02: Standard derivation of data beta_02
    % std_20: Standard derivation of data beta_20
    %%%%% Output %%%%
    % output: The asymptotic variance matrix V/n, where V is defined in equation (29)
    
    % A function to calculate the sum of Omega_j + Omega_j' for all j >= 1
    function Omegaj = element(mag, j)
       Sw = zeros(size(mag, 1), size(mag, 1));
       for i = (j + 1) : size(mag, 2)
           Sw = Sw + mag(:, i)*mag(:, i - j)'; % See the equation below (C2)
       end
       Sw = Sw/size(mag, 2);
       Omegaj = Sw + Sw';
    end
                               
    % Calculate \hat{G}
    type = string(type);
    if (type == 'exact')
        G_hat = zeros(4, 4);
        for t = 1 : size(v, 2)
            G_hat = G_hat + moment_derivatives('exact', r(t), d(t), param, v(t), ...
                                                std_01, std_10, std_11, std_02, std_20); % See equation (C1)
        end
    end
    if (type == 'over')
        G_hat = zeros(4, 5);
        for t = 1 : size(v, 2)
            G_hat = G_hat + moment_derivatives('over', r(t), d(t), param, v(t), ...
                                                std_01, std_10, std_11, std_02, std_20); % See equation (C1)
        end
    end
    G_hat = G_hat/size(v, 2); % See equation (C1)

    % Calculate \hat{\Omega}
    if (type == 'exact')
        mag = moment_fun('exact_g', r, d, param, v, beta01, beta10, beta11, beta02, beta20, ...
                      std_01, std_10, std_11, std_02, std_20);
    end
    if (type == 'over')
        mag = moment_fun('over_g', r, d, param, v, beta01, beta10, beta11, beta02, beta20, ...
                      std_01, std_10, std_11, std_02, std_20);
    end
    nlag = round(0.75*size(mag, 2)^(1/3) - 1); % Lag \ell in equation (C2)
    % Calculate Omega_0
    S = zeros(size(mag, 1), size(mag, 1));
    for k = 1:size(mag,2)
       S = S + mag(:, k)*mag(:, k)';
    end
    Omega0 = S/size(mag, 2);
    Oemga_hat = Omega0;
    for j = 1 : nlag
       Oemga_hat = Oemga_hat + (nlag + 1 - j)/(nlag + 1)*element(mag, j); % See equation (C2)
    end
    
    % Calculate the asymptotic variance matrix V in (29)
    V_0 = G_hat/Oemga_hat*G_hat'; % See equation (C1)
    output = inv(V_0)/size(v, 2);
end